import { Server } from 'socket.io';

interface LocationUpdate {
  id: string;
  lat: number;
  lng: number;
  city: string;
  country: string;
  views: number;
  ranking: number;
  platform: string;
  trend: 'up' | 'down' | 'stable';
  growth: number;
  timestamp: string;
}

interface LiveFeedItem {
  id: string;
  type: 'new_view' | 'ranking_change' | 'trend_spike' | 'viral_content' | 'engagement_surge';
  location: string;
  message: string;
  timestamp: string;
  importance: 'high' | 'medium' | 'low';
  data?: {
    views?: number;
    ranking?: number;
    growth?: number;
    platform?: string;
  };
}

export const setupSocket = (io: Server) => {
  io.on('connection', (socket) => {
    console.log('Client connected:', socket.id);
    
    // Join map room for live updates
    socket.on('join-map', () => {
      socket.join('map-updates');
      console.log(`Client ${socket.id} joined map updates`);
    });

    // Leave map room
    socket.on('leave-map', () => {
      socket.leave('map-updates');
      console.log(`Client ${socket.id} left map updates`);
    });

    // Handle location updates
    socket.on('location-update', (data: LocationUpdate) => {
      // Broadcast to all clients in map room
      io.to('map-updates').emit('location-update', {
        ...data,
        timestamp: new Date().toISOString()
      });
    });

    // Handle live feed events
    socket.on('live-feed-event', (data: LiveFeedItem) => {
      // Broadcast to all clients in map room
      io.to('map-updates').emit('live-feed-event', {
        ...data,
        id: Date.now().toString(),
        timestamp: new Date().toISOString()
      });
    });

    // Handle messages (existing functionality)
    socket.on('message', (msg: { text: string; senderId: string }) => {
      // Echo: broadcast message only the client who send the message
      socket.emit('message', {
        text: `Echo: ${msg.text}`,
        senderId: 'system',
        timestamp: new Date().toISOString(),
      });
    });

    // Handle disconnect
    socket.on('disconnect', () => {
      console.log('Client disconnected:', socket.id);
    });

    // Send welcome message
    socket.emit('message', {
      text: 'Welcome to WebSocket Echo Server!',
      senderId: 'system',
      timestamp: new Date().toISOString(),
    });
  });

  // Simulate live updates and broadcast to connected clients
  setInterval(() => {
    const rooms = io.sockets.adapter.rooms.get('map-updates');
    if (rooms && rooms.size > 0) {
      // Generate random location update
      const cities = [
        { name: 'New York', lat: 40.7128, lng: -74.0060, country: 'USA' },
        { name: 'Los Angeles', lat: 34.0522, lng: -118.2437, country: 'USA' },
        { name: 'London', lat: 51.5074, lng: -0.1278, country: 'UK' },
        { name: 'Tokyo', lat: 35.6762, lng: 139.6503, country: 'Japan' },
        { name: 'Paris', lat: 48.8566, lng: 2.3522, country: 'France' }
      ];
      
      const platforms = ['Instagram', 'TikTok', 'YouTube', 'Twitter', 'LinkedIn'];
      const randomCity = cities[Math.floor(Math.random() * cities.length)];
      const randomPlatform = platforms[Math.floor(Math.random() * platforms.length)];
      
      const locationUpdate: LocationUpdate = {
        id: Math.random().toString(36).substr(2, 9),
        lat: randomCity.lat,
        lng: randomCity.lng,
        city: randomCity.name,
        country: randomCity.country,
        views: Math.floor(Math.random() * 20000) + 1000,
        ranking: Math.floor(Math.random() * 50) + 1,
        platform: randomPlatform,
        trend: Math.random() > 0.6 ? 'up' : Math.random() > 0.3 ? 'stable' : 'down',
        growth: (Math.random() - 0.5) * 20,
        timestamp: new Date().toISOString()
      };

      io.to('map-updates').emit('location-update', locationUpdate);

      // Occasionally send live feed events
      if (Math.random() > 0.7) {
        const feedTypes: LiveFeedItem['type'][] = ['new_view', 'ranking_change', 'trend_spike', 'viral_content', 'engagement_surge'];
        const messages = {
          new_view: [`Surge in views: +${Math.floor(Math.random() * 1000)} in last 5 minutes`],
          ranking_change: [`Ranking improved on ${randomPlatform}`],
          trend_spike: [`Engagement spike detected: ${Math.floor(Math.random() * 300) + 100}% increase`],
          viral_content: ['Content going viral'],
          engagement_surge: [`Engagement rate: ${(Math.random() * 20 + 5).toFixed(1)}%`]
        };
        
        const feedType = feedTypes[Math.floor(Math.random() * feedTypes.length)];
        const feedMessage = messages[feedType][0];
        
        const liveFeedEvent: LiveFeedItem = {
          id: Date.now().toString(),
          type: feedType,
          location: `${randomCity.name}, ${randomCity.country}`,
          message: feedMessage,
          timestamp: new Date().toISOString(),
          importance: Math.random() > 0.8 ? 'high' : Math.random() > 0.5 ? 'medium' : 'low',
          data: {
            platform: randomPlatform,
            views: feedType === 'new_view' ? Math.floor(Math.random() * 1000) : undefined,
            growth: Math.floor(Math.random() * 100) - 50
          }
        };

        io.to('map-updates').emit('live-feed-event', liveFeedEvent);
      }
    }
  }, 8000); // Send updates every 8 seconds
};