'use client'

import { useEffect, useRef, useState } from 'react'
import { io, Socket } from 'socket.io-client'

interface LocationUpdate {
  id: string;
  lat: number;
  lng: number;
  city: string;
  country: string;
  views: number;
  ranking: number;
  platform: string;
  trend: 'up' | 'down' | 'stable';
  growth: number;
  timestamp: string;
}

interface LiveFeedItem {
  id: string;
  type: 'new_view' | 'ranking_change' | 'trend_spike' | 'viral_content' | 'engagement_surge';
  location: string;
  message: string;
  timestamp: string;
  importance: 'high' | 'medium' | 'low';
  data?: {
    views?: number;
    ranking?: number;
    growth?: number;
    platform?: string;
  };
}

export function useWebSocket(room: string = 'map-updates') {
  const [isConnected, setIsConnected] = useState(false)
  const [locationUpdate, setLocationUpdate] = useState<LocationUpdate | null>(null)
  const [liveFeedEvent, setLiveFeedEvent] = useState<LiveFeedItem | null>(null)
  const socketRef = useRef<Socket | null>(null)

  useEffect(() => {
    // Initialize socket connection
    socketRef.current = io()

    const socket = socketRef.current

    socket.on('connect', () => {
      console.log('Connected to WebSocket server')
      setIsConnected(true)
      
      // Join the specified room
      socket.emit('join-map')
    })

    socket.on('disconnect', () => {
      console.log('Disconnected from WebSocket server')
      setIsConnected(false)
    })

    // Listen for location updates
    socket.on('location-update', (data: LocationUpdate) => {
      console.log('Received location update:', data)
      setLocationUpdate(data)
    })

    // Listen for live feed events
    socket.on('live-feed-event', (data: LiveFeedItem) => {
      console.log('Received live feed event:', data)
      setLiveFeedEvent(data)
    })

    // Handle connection errors
    socket.on('connect_error', (error) => {
      console.error('WebSocket connection error:', error)
      setIsConnected(false)
    })

    return () => {
      // Leave room and disconnect on cleanup
      if (socketRef.current) {
        socketRef.current.emit('leave-map')
        socketRef.current.disconnect()
      }
    }
  }, [room])

  // Function to send location updates
  const sendLocationUpdate = (data: Omit<LocationUpdate, 'id' | 'timestamp'>) => {
    if (socketRef.current && isConnected) {
      socketRef.current.emit('location-update', data)
    }
  }

  // Function to send live feed events
  const sendLiveFeedEvent = (data: Omit<LiveFeedItem, 'id' | 'timestamp'>) => {
    if (socketRef.current && isConnected) {
      socketRef.current.emit('live-feed-event', data)
    }
  }

  return {
    isConnected,
    locationUpdate,
    liveFeedEvent,
    sendLocationUpdate,
    sendLiveFeedEvent
  }
}