'use client'

import { useState, useRef, useEffect } from 'react'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Badge } from '@/components/ui/badge'
import { Card, CardContent } from '@/components/ui/card'
import { 
  Send, 
  Sparkles, 
  Bot, 
  User, 
  Paperclip, 
  Smile, 
  Mic,
  MicOff,
  StopCircle,
  Copy,
  ThumbsUp,
  ThumbsDown,
  RefreshCw,
  Zap
} from 'lucide-react'

interface Message {
  id: string
  role: 'user' | 'assistant'
  content: string
  timestamp: Date
  reactions?: {
    thumbsUp?: number
    thumbsDown?: number
  }
}

interface TypingIndicator {
  isTyping: boolean
  text: string
}

export function ModernChatbot() {
  const [messages, setMessages] = useState<Message[]>([
    {
      id: '1',
      role: 'assistant',
      content: "👋 Hi there! I'm your AI assistant, ready to help you create amazing content, analyze data, and boost your social media presence. What can I help you with today?",
      timestamp: new Date(Date.now() - 60000),
      reactions: { thumbsUp: 2, thumbsDown: 0 }
    }
  ])
  const [input, setInput] = useState('')
  const [isTyping, setIsTyping] = useState(false)
  const [isRecording, setIsRecording] = useState(false)
  const [isGenerating, setIsGenerating] = useState(false)
  const [connectionError, setConnectionError] = useState<string | null>(null)
  const [typingIndicator, setTypingIndicator] = useState<TypingIndicator>({ isTyping: false, text: '' })
  const messagesEndRef = useRef<HTMLDivElement>(null)
  const inputRef = useRef<HTMLInputElement>(null)

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' })
  }

  useEffect(() => {
    scrollToBottom()
  }, [messages])

  const simulateTyping = async () => {
    setTypingIndicator({ isTyping: true, text: 'AI is thinking' })
    
    const typingTexts = ['AI is thinking', 'AI is processing', 'AI is analyzing', 'AI is generating']
    let index = 0
    
    const typingInterval = setInterval(() => {
      setTypingIndicator({ isTyping: true, text: typingTexts[index] + '...' })
      index = (index + 1) % typingTexts.length
    }, 800)

    await new Promise(resolve => setTimeout(resolve, 2000))
    clearInterval(typingInterval)
    setTypingIndicator({ isTyping: false, text: '' })
  }

  const generateAIResponse = async (userMessage: string): Promise<string> => {
    try {
      setConnectionError(null)
      console.log('Sending AI request:', userMessage)
      
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'content',
          prompt: userMessage,
          options: {
            contentType: 'chat response',
            tone: 'helpful',
            length: 'short'
          }
        })
      })

      console.log('AI Response status:', response.status)

      if (!response.ok) {
        const errorText = await response.text()
        console.error('API Error Response:', errorText)
        setConnectionError(`Service error: ${response.status}`)
        throw new Error(`API Error: ${response.status} - ${errorText}`)
      }

      const data = await response.json()
      console.log('AI Response data:', data)
      
      if (data.error) {
        console.error('AI returned error:', data.error)
        setConnectionError(data.error)
        throw new Error(data.error)
      }

      if (!data.content) {
        console.warn('No content in AI response, using fallback')
        return "I'm here to help! What would you like to know?"
      }

      setConnectionError(null)
      return data.content
    } catch (error) {
      console.error('Error generating AI response:', error)
      
      // Show more specific error messages
      let errorMessage = "I'm sorry, I encountered an error. Please try again!"
      
      if (error instanceof Error) {
        if (error.message.includes('Failed to fetch')) {
          errorMessage = "Network error: Please check your connection and try again."
          setConnectionError("Network connection failed")
        } else if (error.message.includes('API Error')) {
          errorMessage = "Service temporarily unavailable. Please try again in a moment."
          setConnectionError("AI service unavailable")
        } else {
          errorMessage = `Error: ${error.message}`
          setConnectionError(error.message)
        }
      }
      
      // Fallback to mock responses if API fails
      const fallbackResponses = [
        "That's a great question! Based on my analysis, I recommend focusing on creating engaging content that resonates with your target audience. 🎯",
        "I can help you with that! Let me break this down into actionable steps that will drive results for your social media strategy. 📈",
        "Excellent point! Here's what I suggest: leverage AI-powered tools to optimize your content and track performance metrics in real-time. 🤖",
        "I love your enthusiasm! Let's explore some creative approaches that will make your content stand out and drive engagement. ✨",
        "Great insight! Based on current trends, I recommend incorporating video content and interactive elements to boost engagement. 🎥"
      ]
      
      // Return error message first, then fallback on subsequent attempts
      return Math.random() > 0.5 ? errorMessage : fallbackResponses[Math.floor(Math.random() * fallbackResponses.length)]
    }
  }

  const handleSend = async () => {
    if (!input.trim()) return

    const userMessage: Message = {
      id: Date.now().toString(),
      role: 'user',
      content: input,
      timestamp: new Date()
    }

    setMessages(prev => [...prev, userMessage])
    setInput('')
    setIsTyping(true)
    setIsGenerating(true)

    // Simulate AI typing
    await simulateTyping()

    try {
      const aiResponse = await generateAIResponse(input)
      const assistantMessage: Message = {
        id: (Date.now() + 1).toString(),
        role: 'assistant',
        content: aiResponse,
        timestamp: new Date(),
        reactions: { thumbsUp: 0, thumbsDown: 0 }
      }

      setMessages(prev => [...prev, assistantMessage])
    } catch (error) {
      const errorMessage: Message = {
        id: (Date.now() + 1).toString(),
        role: 'assistant',
        content: "Sorry, I encountered an error. Please try again!",
        timestamp: new Date()
      }
      setMessages(prev => [...prev, errorMessage])
    } finally {
      setIsTyping(false)
      setIsGenerating(false)
    }
  }

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault()
      handleSend()
    }
  }

  const handleReaction = (messageId: string, reaction: 'thumbsUp' | 'thumbsDown') => {
    setMessages(prev => prev.map(msg => {
      if (msg.id === messageId) {
        return {
          ...msg,
          reactions: {
            ...msg.reactions,
            [reaction]: (msg.reactions?.[reaction] || 0) + 1
          }
        }
      }
      return msg
    }))
  }

  const copyMessage = (content: string) => {
    navigator.clipboard.writeText(content)
    // You could add a toast notification here
  }

  const regenerateResponse = async (messageId: string) => {
    const messageIndex = messages.findIndex(msg => msg.id === messageId)
    if (messageIndex === -1) return

    const previousMessage = messages[messageIndex - 1]
    if (!previousMessage || previousMessage.role !== 'user') return

    setIsTyping(true)
    setIsGenerating(true)
    await simulateTyping()

    try {
      const newResponse = await generateAIResponse(previousMessage.content + " (Please provide a different response)")
      const newMessage: Message = {
        id: Date.now().toString(),
        role: 'assistant',
        content: newResponse,
        timestamp: new Date(),
        reactions: { thumbsUp: 0, thumbsDown: 0 }
      }

      setMessages(prev => {
        const newMessages = [...prev]
        newMessages[messageIndex] = newMessage
        return newMessages
      })
    } catch (error) {
      console.error('Error regenerating response:', error)
      const errorMessage: Message = {
        id: Date.now().toString(),
        role: 'assistant',
        content: "Sorry, I encountered an error while regenerating the response. Please try again!",
        timestamp: new Date()
      }
      setMessages(prev => {
        const newMessages = [...prev]
        newMessages[messageIndex] = errorMessage
        return newMessages
      })
    } finally {
      setIsTyping(false)
      setIsGenerating(false)
    }
  }

  const formatTime = (date: Date) => {
    return date.toLocaleTimeString('en-US', { 
      hour: '2-digit', 
      minute: '2-digit',
      hour12: true 
    })
  }

  return (
    <div className="relative h-full min-h-[600px] rounded-2xl overflow-hidden">
      {/* Background Image with Overlay */}
      <div 
        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
        style={{ backgroundImage: 'url(/chatbot-bg.jpg)' }}
      >
        <div className="absolute inset-0 bg-gradient-to-br from-purple-900/80 via-blue-900/70 to-pink-900/60 backdrop-blur-sm"></div>
      </div>

      {/* Chat Container */}
      <div className="relative h-full flex flex-col">
        {/* Header */}
        <div className="relative z-10 p-6 border-b border-white/10">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <div className="relative">
                <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center">
                  <Bot className="w-6 h-6 text-white" />
                </div>
                <div className={`absolute -bottom-1 -right-1 w-4 h-4 rounded-full border-2 border-white ${
                  connectionError ? 'bg-red-500' : isGenerating ? 'bg-yellow-500 animate-pulse' : 'bg-green-500'
                }`}></div>
              </div>
              <div>
                <h2 className="text-xl font-bold text-white">AI Assistant</h2>
                <div className="flex items-center space-x-2">
                  <div className={`w-2 h-2 rounded-full ${
                    connectionError ? 'bg-red-500' : isGenerating ? 'bg-yellow-500 animate-pulse' : 'bg-green-500 animate-pulse'
                  }`}></div>
                  <span className="text-sm text-white/80">
                    {connectionError ? 'Connection Error' : isGenerating ? 'Processing...' : 'Online • Ready to help'}
                  </span>
                </div>
              </div>
            </div>
            <div className="flex items-center space-x-2">
              {connectionError && (
                <Badge variant="destructive" className="bg-red-500 text-white border-0">
                  {connectionError}
                </Badge>
              )}
              <Badge className="bg-gradient-to-r from-purple-500 to-pink-500 text-white border-0">
                <Sparkles className="w-3 h-3 mr-1" />
                AI Powered
              </Badge>
              <Badge variant="outline" className="border-white/20 text-white/80">
                <Zap className="w-3 h-3 mr-1" />
                GPT-4
              </Badge>
            </div>
          </div>
        </div>

        {/* Messages Area */}
        <div className="flex-1 overflow-y-auto p-6 space-y-4">
          {connectionError && (
                <div className="p-3 bg-red-500/10 border border-red-500/30 rounded-lg mb-4">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-2">
                      <div className="w-4 h-4 bg-red-500 rounded-full animate-pulse"></div>
                      <span className="text-sm text-red-200">Connection issue detected</span>
                    </div>
                    <Button
                      size="sm"
                      variant="ghost"
                      className="h-6 px-2 text-xs text-red-200 hover:text-white hover:bg-red-500/20"
                      onClick={() => {
                        setConnectionError(null)
                        handleSend()
                      }}
                    >
                      Retry
                    </Button>
                  </div>
                </div>
              )}
              
              {messages.map((message) => (
            <div
              key={message.id}
              className={`flex ${message.role === 'user' ? 'justify-end' : 'justify-start'} animate-fade-in`}
            >
              <div className={`max-w-lg ${message.role === 'user' ? 'order-2' : 'order-1'}`}>
                <div className="flex items-end space-x-2 mb-1">
                  {message.role === 'assistant' && (
                    <div className="w-8 h-8 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center flex-shrink-0">
                      <Bot className="w-4 h-4 text-white" />
                    </div>
                  )}
                  <div className="text-xs text-white/60">
                    {formatTime(message.timestamp)}
                  </div>
                  {message.role === 'user' && (
                    <div className="w-8 h-8 bg-gradient-to-r from-green-500 to-blue-500 rounded-full flex items-center justify-center flex-shrink-0">
                      <User className="w-4 h-4 text-white" />
                    </div>
                  )}
                </div>
                
                <Card className={`${
                  message.role === 'user' 
                    ? 'bg-gradient-to-r from-blue-600 to-purple-600 text-white border-0 shadow-lg' 
                    : 'bg-white/10 backdrop-blur-md text-white border border-white/20 shadow-xl'
                }`}>
                  <CardContent className="p-4">
                    <p className="text-sm leading-relaxed whitespace-pre-wrap">{message.content}</p>
                    
                    {/* Message Actions */}
                    <div className="flex items-center justify-between mt-3 pt-3 border-t border-white/10">
                      <div className="flex items-center space-x-2">
                        {message.role === 'assistant' && (
                          <>
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-6 px-2 text-white/60 hover:text-white hover:bg-white/10"
                              onClick={() => copyMessage(message.content)}
                            >
                              <Copy className="w-3 h-3" />
                            </Button>
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-6 px-2 text-white/60 hover:text-white hover:bg-white/10"
                              onClick={() => regenerateResponse(message.id)}
                            >
                              <RefreshCw className="w-3 h-3" />
                            </Button>
                          </>
                        )}
                      </div>
                      
                      {message.role === 'assistant' && message.reactions && (
                        <div className="flex items-center space-x-1">
                          <Button
                            size="sm"
                            variant="ghost"
                            className="h-6 px-2 text-white/60 hover:text-green-400 hover:bg-white/10"
                            onClick={() => handleReaction(message.id, 'thumbsUp')}
                          >
                            <ThumbsUp className="w-3 h-3" />
                            {message.reactions.thumbsUp > 0 && (
                              <span className="text-xs ml-1">{message.reactions.thumbsUp}</span>
                            )}
                          </Button>
                          <Button
                            size="sm"
                            variant="ghost"
                            className="h-6 px-2 text-white/60 hover:text-red-400 hover:bg-white/10"
                            onClick={() => handleReaction(message.id, 'thumbsDown')}
                          >
                            <ThumbsDown className="w-3 h-3" />
                            {message.reactions.thumbsDown > 0 && (
                              <span className="text-xs ml-1">{message.reactions.thumbsDown}</span>
                            )}
                          </Button>
                        </div>
                      )}
                    </div>
                  </CardContent>
                </Card>
              </div>
            </div>
          ))}

          {/* Typing Indicator */}
          {typingIndicator.isTyping && (
            <div className="flex justify-start animate-fade-in">
              <div className="max-w-lg">
                <div className="flex items-center space-x-2 mb-1">
                  <div className="w-8 h-8 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center">
                    <Bot className="w-4 h-4 text-white" />
                  </div>
                </div>
                <Card className="bg-white/10 backdrop-blur-md text-white border border-white/20">
                  <CardContent className="p-4">
                    <div className="flex items-center space-x-2">
                      <div className="flex space-x-1">
                        <div className="w-2 h-2 bg-white rounded-full animate-bounce" style={{ animationDelay: '0ms' }}></div>
                        <div className="w-2 h-2 bg-white rounded-full animate-bounce" style={{ animationDelay: '150ms' }}></div>
                        <div className="w-2 h-2 bg-white rounded-full animate-bounce" style={{ animationDelay: '300ms' }}></div>
                      </div>
                      <span className="text-sm text-white/80">{typingIndicator.text}</span>
                    </div>
                  </CardContent>
                </Card>
              </div>
            </div>
          )}

          <div ref={messagesEndRef} />
        </div>

        {/* Input Area */}
        <div className="relative z-10 p-6 border-t border-white/10">
          <div className="relative">
            <div className="flex items-center space-x-2">
              <Button
                size="sm"
                variant="ghost"
                className="text-white/60 hover:text-white hover:bg-white/10"
              >
                <Paperclip className="w-4 h-4" />
              </Button>
              
              <div className="flex-1 relative">
                <Input
                  ref={inputRef}
                  value={input}
                  onChange={(e) => setInput(e.target.value)}
                  onKeyPress={handleKeyPress}
                  placeholder="Type your message..."
                  className="bg-white/10 border-white/20 text-white placeholder-white/50 focus:border-white/40 focus:ring-white/20 pr-12"
                  disabled={isTyping}
                />
                <Button
                  size="sm"
                  variant="ghost"
                  className="absolute right-2 top-1/2 transform -translate-y-1/2 text-white/60 hover:text-white hover:bg-white/10"
                >
                  <Smile className="w-4 h-4" />
                </Button>
              </div>

              <Button
                size="sm"
                variant="ghost"
                className={`text-white/60 hover:text-white hover:bg-white/10 ${
                  isRecording ? 'text-red-400 hover:text-red-400' : ''
                }`}
                onClick={() => setIsRecording(!isRecording)}
              >
                {isRecording ? <MicOff className="w-4 h-4" /> : <Mic className="w-4 h-4" />}
              </Button>

              <Button
                size="sm"
                onClick={handleSend}
                disabled={!input.trim() || isTyping}
                className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white border-0 shadow-lg disabled:opacity-50"
              >
                {isGenerating ? (
                  <div className="w-4 h-4 border border-white/30 border-t-white rounded-full animate-spin"></div>
                ) : isTyping ? (
                  <StopCircle className="w-4 h-4" />
                ) : (
                  <Send className="w-4 h-4" />
                )}
              </Button>
            </div>
          </div>
          
          {/* Quick Actions */}
          <div className="flex items-center justify-between mt-3">
            <div className="flex items-center space-x-2">
              <span className="text-xs text-white/50">Quick actions:</span>
              <Button
                size="sm"
                variant="ghost"
                className="h-6 px-2 text-xs text-white/50 hover:text-white hover:bg-white/10"
                onClick={() => setInput("Help me create a social media post")}
              >
                Create Post
              </Button>
              <Button
                size="sm"
                variant="ghost"
                className="h-6 px-2 text-xs text-white/50 hover:text-white hover:bg-white/10"
                onClick={() => setInput("Analyze my content performance")}
              >
                Analyze
              </Button>
              <Button
                size="sm"
                variant="ghost"
                className="h-6 px-2 text-xs text-white/50 hover:text-white hover:bg-white/10"
                onClick={() => setInput("Generate content ideas")}
              >
                Ideas
              </Button>
              <Button
                size="sm"
                className="h-6 px-3 text-xs bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700 text-white border-0 shadow-md disabled:opacity-50"
                onClick={async () => {
                  const aiPrompt = "Generate creative content ideas for my social media"
                  setInput(aiPrompt)
                  setTimeout(() => handleSend(), 100)
                }}
                disabled={isTyping || isGenerating}
              >
                {isGenerating ? (
                  <>
                    <div className="w-3 h-3 border border-white/30 border-t-white rounded-full animate-spin mr-1"></div>
                    Generating...
                  </>
                ) : (
                  <>
                    <Sparkles className="w-3 h-3 mr-1" />
                    AI Generate
                  </>
                )}
              </Button>
            </div>
            <span className="text-xs text-white/50">
              Press Enter to send • Shift+Enter for new line
            </span>
          </div>
        </div>
      </div>

      <style jsx>{`
        @keyframes fade-in {
          from {
            opacity: 0;
            transform: translateY(10px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
        
        .animate-fade-in {
          animation: fade-in 0.3s ease-out;
        }
      `}</style>
    </div>
  )
}