'use client'

import { useState } from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Textarea } from '@/components/ui/textarea'
import { Badge } from '@/components/ui/badge'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Label } from '@/components/ui/label'
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog'
import { 
  Upload, 
  Image as ImageIcon, 
  Video, 
  FileText, 
  Calendar,
  Send,
  Plus,
  Instagram,
  Twitter,
  Facebook,
  Linkedin,
  TrendingUp,
  Users,
  Heart,
  MessageCircle,
  Share,
  X,
  Music,
  Brain,
  Clock,
  BarChart3,
  Zap,
  Target,
  AlertCircle,
  Check,
  Sparkles
} from 'lucide-react'

export function SocialMediaTab() {
  const [selectedPlatforms, setSelectedPlatforms] = useState<string[]>([])
  const [contentType, setContentType] = useState<'text' | 'image' | 'video' | 'article'>('text')
  const [content, setContent] = useState('')
  const [mediaFiles, setMediaFiles] = useState<File[]>([])
  const [isAccountModalOpen, setIsAccountModalOpen] = useState(false)
  const [selectedPlatform, setSelectedPlatform] = useState('')
  const [accountUsername, setAccountUsername] = useState('')
  const [accountPassword, setAccountPassword] = useState('')
  
  // Trends state
  const [isAnalyzingTrends, setIsAnalyzingTrends] = useState(false)
  const [trendsData, setTrendsData] = useState<any>(null)
  const [trendsError, setTrendsError] = useState('')
  const [selectedTimeframe, setSelectedTimeframe] = useState('24h')
  const [selectedCategory, setSelectedCategory] = useState('general')
  const [selectedTrendPlatforms, setSelectedTrendPlatforms] = useState(['instagram', 'twitter', 'tiktok'])
  const [generatedContent, setGeneratedContent] = useState<any>(null)
  const [isGeneratingContent, setIsGeneratingContent] = useState(false)
  const [contentError, setContentError] = useState('')
  const [aiPrompt, setAiPrompt] = useState('')
  const [contentTone, setContentTone] = useState('professional')

  const platforms = [
    { id: 'instagram', name: 'Instagram', icon: Instagram, color: 'bg-gradient-to-r from-purple-600 to-pink-600' },
    { id: 'twitter', name: 'Twitter', icon: Twitter, color: 'bg-blue-400' },
    { id: 'facebook', name: 'Facebook', icon: Facebook, color: 'bg-blue-600' },
    { id: 'linkedin', name: 'LinkedIn', icon: Linkedin, color: 'bg-blue-700' },
    { id: 'tiktok', name: 'TikTok', icon: Music, color: 'bg-gradient-to-r from-black to-gray-800' }
  ]

  const socialStats = [
    {
      platform: 'Instagram',
      followers: '45.2K',
      engagement: '6.8%',
      growth: '+12.5%',
      posts: 156,
      icon: Instagram,
      color: 'from-purple-600 to-pink-600'
    },
    {
      platform: 'Twitter',
      followers: '22.8K',
      engagement: '4.2%',
      growth: '+8.3%',
      posts: 342,
      icon: Twitter,
      color: 'from-blue-400 to-blue-500'
    },
    {
      platform: 'Facebook',
      followers: '38.5K',
      engagement: '5.1%',
      growth: '+6.7%',
      posts: 89,
      icon: Facebook,
      color: 'from-blue-600 to-blue-700'
    },
    {
      platform: 'LinkedIn',
      followers: '18.0K',
      engagement: '8.9%',
      growth: '+15.2%',
      posts: 67,
      icon: Linkedin,
      color: 'from-blue-700 to-blue-800'
    },
    {
      platform: 'TikTok',
      followers: '67.3K',
      engagement: '12.4%',
      growth: '+28.9%',
      posts: 234,
      icon: Music,
      color: 'from-black to-gray-800'
    }
  ]

  const recentPosts = [
    {
      id: 1,
      platform: 'Instagram',
      content: 'Check out our latest product launch! 🚀 #innovation #tech',
      image: true,
      likes: 1234,
      comments: 89,
      shares: 45,
      postedAt: '2 hours ago',
      performance: 'High'
    },
    {
      id: 2,
      platform: 'Twitter',
      content: 'Excited to announce our new partnership with TechCorp! This collaboration will bring amazing value to our customers.',
      image: false,
      likes: 567,
      comments: 23,
      shares: 78,
      postedAt: '4 hours ago',
      performance: 'Medium'
    },
    {
      id: 3,
      platform: 'LinkedIn',
      content: 'Industry insights: The future of AI in social media marketing is here. Read our latest article to learn more.',
      image: true,
      likes: 234,
      comments: 45,
      shares: 123,
      postedAt: '6 hours ago',
      performance: 'High'
    },
    {
      id: 4,
      platform: 'TikTok',
      content: 'New trend alert! Watch us demonstrate our latest feature in under 60 seconds ⏱️ #tech #viral #fyp',
      image: true,
      likes: 5678,
      comments: 234,
      shares: 189,
      postedAt: '8 hours ago',
      performance: 'High'
    }
  ]

  const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(event.target.files || [])
    setMediaFiles(prev => [...prev, ...files])
  }

  const removeFile = (index: number) => {
    setMediaFiles(prev => prev.filter((_, i) => i !== index))
  }

  const togglePlatform = (platformId: string) => {
    setSelectedPlatforms(prev => 
      prev.includes(platformId) 
        ? prev.filter(p => p !== platformId)
        : [...prev, platformId]
    )
  }

  const handleConnectAccount = () => {
    setIsAccountModalOpen(true)
  }

  const handleAccountSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    
    if (!selectedPlatform || !accountUsername) {
      alert('Please select a platform and enter a username')
      return
    }

    try {
      // Here you would make an API call to connect the account
      console.log('Connecting account:', {
        platform: selectedPlatform,
        username: accountUsername,
        password: accountPassword ? '***' : ''
      })

      // Simulate API call
      alert(`Successfully connected ${selectedPlatform} account: @${accountUsername}`)
      
      // Reset form
      setSelectedPlatform('')
      setAccountUsername('')
      setAccountPassword('')
      setIsAccountModalOpen(false)
    } catch (error) {
      console.error('Error connecting account:', error)
      alert('Failed to connect account. Please try again.')
    }
  }

  const handlePostNow = () => {
    if (selectedPlatforms.length === 0) {
      alert('Please select at least one platform to post to')
      return
    }
    if (!content.trim()) {
      alert('Please enter some content to post')
      return
    }
    
    alert(`Post would be published to: ${selectedPlatforms.join(', ')}\n\nContent: ${content}`)
  }

  const handleGenerateContent = async () => {
    if (!aiPrompt.trim()) {
      setContentError('Please enter what you want to create')
      return
    }
    
    if (selectedPlatforms.length === 0) {
      setContentError('Please select at least one platform')
      return
    }

    setIsGeneratingContent(true)
    setContentError('')
    setGeneratedContent(null)

    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'social-media-content',
          prompt: aiPrompt,
          options: {
            platforms: selectedPlatforms,
            tone: contentTone,
            contentType: contentType,
            category: selectedCategory
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to generate content')
      }

      const data = await response.json()
      
      console.log('AI Content Generation Response:', data)
      
      if (data.content) {
        setGeneratedContent(data.content)
        setContent(data.content.content || '')
      } else {
        setContentError('No content generated. Please try again.')
      }
    } catch (error) {
      console.error('Error generating content:', error)
      setContentError('Failed to generate content. Please try again.')
    } finally {
      setIsGeneratingContent(false)
    }
  }

  const handleStatCardClick = (platform: string) => {
    alert(`Detailed ${platform} analytics coming soon! This will show follower growth, engagement metrics, and post performance.`)
  }

  const handleSchedule = () => {
    if (selectedPlatforms.length === 0) {
      alert('Please select at least one platform to schedule for')
      return
    }
    if (!content.trim()) {
      alert('Please enter some content to schedule')
      return
    }
    
    alert(`Post would be scheduled for: ${selectedPlatforms.join(', ')}\n\nContent: ${content}`)
  }

  const analyzeTrends = async () => {
    setIsAnalyzingTrends(true)
    setTrendsError('')
    setTrendsData(null)

    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'trends',
          prompt: `Analyze social media trends for ${selectedCategory} category`,
          options: {
            platforms: selectedTrendPlatforms,
            timeframe: selectedTimeframe,
            category: selectedCategory
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to analyze trends')
      }

      const data = await response.json()
      
      if (data.trends) {
        setTrendsData(data.trends)
      } else {
        setTrendsError('No trends data received')
      }
    } catch (error) {
      console.error('Error analyzing trends:', error)
      setTrendsError('Failed to analyze trends. Please try again.')
    } finally {
      setIsAnalyzingTrends(false)
    }
  }

  const generateContentFromTrend = async (trend: any) => {
    setIsGeneratingContent(true)
    setContentError('')
    setGeneratedContent(null)

    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'trending-content',
          prompt: `Create content based on trend: ${trend.topic}`,
          options: {
            trend: trend.topic,
            platform: selectedPlatforms[0] || 'instagram',
            contentType: contentType,
            tone: 'engaging'
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to generate content')
      }

      const data = await response.json()
      
      if (data.content) {
        setGeneratedContent(data.content)
        setContent(data.content.content || '')
      } else {
        setContentError('Failed to generate content')
      }
    } catch (error) {
      console.error('Error generating content:', error)
      setContentError('Failed to generate content. Please try again.')
    } finally {
      setIsGeneratingContent(false)
    }
  }

  const toggleTrendPlatform = (platformId: string) => {
    setSelectedTrendPlatforms(prev => 
      prev.includes(platformId) 
        ? prev.filter(p => p !== platformId)
        : [...prev, platformId]
    )
  }

  const getEngagementColor = (engagement: string) => {
    switch (engagement.toLowerCase()) {
      case 'high': return 'text-green-600 bg-green-50'
      case 'medium': return 'text-yellow-600 bg-yellow-50'
      case 'low': return 'text-red-600 bg-red-50'
      default: return 'text-gray-600 bg-gray-50'
    }
  }

  const getSentimentColor = (sentiment: string) => {
    switch (sentiment.toLowerCase()) {
      case 'positive': return 'text-green-600'
      case 'neutral': return 'text-gray-600'
      case 'negative': return 'text-red-600'
      default: return 'text-gray-600'
    }
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Social Media Management</h1>
          <p className="text-gray-600 mt-2">Manage your social media accounts and content</p>
        </div>
        <Dialog open={isAccountModalOpen} onOpenChange={setIsAccountModalOpen}>
          <DialogTrigger asChild>
            <Button 
              onClick={handleConnectAccount}
              className="hover:scale-105 transition-transform duration-200"
            >
              <Plus className="mr-2 h-4 w-4" />
              Connect Account
            </Button>
          </DialogTrigger>
          <DialogContent className="sm:max-w-[425px]">
            <DialogHeader>
              <DialogTitle>Connect Social Media Account</DialogTitle>
              <DialogDescription>
                Add a new social media account to manage your content and analytics.
              </DialogDescription>
            </DialogHeader>
            <form onSubmit={handleAccountSubmit} className="space-y-4">
              <div>
                <Label htmlFor="platform">Platform</Label>
                <Select value={selectedPlatform} onValueChange={setSelectedPlatform}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select a platform" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="instagram">Instagram</SelectItem>
                    <SelectItem value="twitter">Twitter</SelectItem>
                    <SelectItem value="facebook">Facebook</SelectItem>
                    <SelectItem value="linkedin">LinkedIn</SelectItem>
                    <SelectItem value="tiktok">TikTok</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              <div>
                <Label htmlFor="username">Username/Email</Label>
                <Input
                  id="username"
                  type="text"
                  placeholder="Enter your username or email"
                  value={accountUsername}
                  onChange={(e) => setAccountUsername(e.target.value)}
                  required
                />
              </div>
              
              <div>
                <Label htmlFor="password">Password</Label>
                <Input
                  id="password"
                  type="password"
                  placeholder="Enter your password"
                  value={accountPassword}
                  onChange={(e) => setAccountPassword(e.target.value)}
                />
              </div>
              
              <div className="flex justify-end space-x-3 pt-4">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => setIsAccountModalOpen(false)}
                >
                  Cancel
                </Button>
                <Button type="submit">
                  Connect Account
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      {/* Platform Stats */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-5 gap-4 sm:gap-6">
        {socialStats.map((stat, index) => {
          const Icon = stat.icon
          return (
            <Card 
              key={index} 
              className="cursor-pointer hover:shadow-lg transition-shadow duration-200"
              onClick={() => handleStatCardClick(stat.platform)}
            >
              <CardContent className="p-4 sm:p-6">
                <div className="flex items-center justify-between mb-3 sm:mb-4">
                  <div className={`w-10 h-10 sm:w-12 sm:h-12 bg-gradient-to-r ${stat.color} rounded-lg flex items-center justify-center`}>
                    <Icon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                  </div>
                  <Badge variant="secondary" className="text-green-600 text-xs">
                    {stat.growth}
                  </Badge>
                </div>
                <h3 className="font-semibold text-gray-900 text-sm sm:text-base">{stat.platform}</h3>
                <p className="text-xl sm:text-2xl font-bold text-gray-900 mt-1">{stat.followers}</p>
                <div className="flex items-center justify-between mt-2 sm:mt-3 text-xs text-gray-600">
                  <span>{stat.engagement} engagement</span>
                  <span>{stat.posts} posts</span>
                </div>
              </CardContent>
            </Card>
          )
        })}
      </div>

      <Tabs defaultValue="create" className="space-y-6">
        <div className="w-full overflow-x-auto">
          <TabsList className="w-full justify-start sm:justify-center min-w-max">
            <TabsTrigger value="create" className="text-xs sm:text-sm">Create Content</TabsTrigger>
            <TabsTrigger value="trends" className="text-xs sm:text-sm">Trending Algorithm</TabsTrigger>
            <TabsTrigger value="posts" className="text-xs sm:text-sm">Recent Posts</TabsTrigger>
            <TabsTrigger value="analytics" className="text-xs sm:text-sm">Analytics</TabsTrigger>
            <TabsTrigger value="scheduler" className="text-xs sm:text-sm">Content Scheduler</TabsTrigger>
          </TabsList>
        </div>

        <TabsContent value="create">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-4 sm:gap-6">
            {/* Content Creation */}
            <Card>
              <CardHeader className="pb-4">
                <CardTitle className="text-lg sm:text-xl">Create New Post</CardTitle>
                <CardDescription className="text-sm">Design and schedule your social media content</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4 sm:space-y-6">
                {/* Platform Selection */}
                <div>
                  <Label className="text-sm font-medium text-gray-700 mb-2 sm:mb-3 block">Select Platforms</Label>
                  <div className="grid grid-cols-2 sm:grid-cols-3 gap-2 sm:gap-3">
                    {platforms.map((platform) => {
                      const Icon = platform.icon
                      return (
                        <Button
                          key={platform.id}
                          variant={selectedPlatforms.includes(platform.id) ? "default" : "outline"}
                          className={`justify-start text-xs sm:text-sm ${
                            selectedPlatforms.includes(platform.id) 
                              ? `bg-gradient-to-r ${platform.color} text-white` 
                              : ''
                          }`}
                          onClick={() => togglePlatform(platform.id)}
                        >
                          <Icon className="mr-1 sm:mr-2 h-3 w-3 sm:h-4 sm:w-4" />
                          <span className="hidden sm:inline">{platform.name}</span>
                          <span className="sm:hidden">{platform.name.slice(0, 4)}</span>
                        </Button>
                      )
                    })}
                  </div>
                </div>

                {/* Content Type */}
                <div>
                  <Label className="text-sm font-medium text-gray-700 mb-2 sm:mb-3 block">Content Type</Label>
                  <div className="grid grid-cols-4 gap-2 sm:gap-3">
                    <Button
                      variant={contentType === 'text' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setContentType('text')}
                      className="h-8 sm:h-10"
                    >
                      <FileText className="h-3 w-3 sm:h-4 sm:w-4" />
                    </Button>
                    <Button
                      variant={contentType === 'image' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setContentType('image')}
                      className="h-8 sm:h-10"
                    >
                      <ImageIcon className="h-3 w-3 sm:h-4 sm:w-4" />
                    </Button>
                    <Button
                      variant={contentType === 'video' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setContentType('video')}
                      className="h-8 sm:h-10"
                    >
                      <Video className="h-3 w-3 sm:h-4 sm:w-4" />
                    </Button>
                    <Button
                      variant={contentType === 'article' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setContentType('article')}
                      className="h-8 sm:h-10"
                    >
                      <FileText className="h-3 w-3 sm:h-4 sm:w-4" />
                    </Button>
                  </div>
                </div>

                {/* Content Input */}
                <div>
                  <Label htmlFor="content" className="text-sm font-medium text-gray-700 mb-2 block">
                    Content
                  </Label>
                  <Textarea
                    id="content"
                    placeholder="Write your post content here..."
                    value={content}
                    onChange={(e) => setContent(e.target.value)}
                    rows={4}
                    className="text-sm"
                  />
                </div>

                {/* Media Upload */}
                {(contentType === 'image' || contentType === 'video') && (
                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">
                      Upload {contentType === 'image' ? 'Images' : 'Videos'}
                    </Label>
                    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                      <input
                        type="file"
                        multiple
                        accept={contentType === 'image' ? 'image/*' : 'video/*'}
                        onChange={handleFileUpload}
                        className="hidden"
                        id="media-upload"
                      />
                      <label htmlFor="media-upload" className="cursor-pointer">
                        <Upload className="h-8 w-8 text-gray-400 mx-auto mb-2" />
                        <p className="text-sm text-gray-600">
                          Click to upload {contentType === 'image' ? 'images' : 'videos'}
                        </p>
                      </label>
                    </div>
                    
                    {mediaFiles.length > 0 && (
                      <div className="mt-4 space-y-2">
                        {mediaFiles.map((file, index) => (
                          <div key={index} className="flex items-center justify-between p-2 bg-gray-50 rounded">
                            <span className="text-sm text-gray-700">{file.name}</span>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => removeFile(index)}
                            >
                              ×
                            </Button>
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                )}

                {/* Action Buttons */}
                <div className="flex space-x-3">
                  <Button 
                    className="flex-1 hover:scale-105 transition-transform duration-200"
                    onClick={handlePostNow}
                  >
                    <Send className="mr-2 h-4 w-4" />
                    Post Now
                  </Button>
                  <Button 
                    variant="outline" 
                    className="flex-1 hover:scale-105 transition-transform duration-200"
                    onClick={handleSchedule}
                  >
                    <Calendar className="mr-2 h-4 w-4" />
                    Schedule
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* AI Content Assistant */}
            <Card>
              <CardHeader>
                <CardTitle>AI Content Assistant</CardTitle>
                <CardDescription>Generate optimized content with AI</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="ai-prompt" className="text-sm font-medium text-gray-700 mb-2 block">
                    What do you want to create?
                  </Label>
                  <Textarea
                    id="ai-prompt"
                    placeholder="E.g., Create an engaging post about our new product launch..."
                    rows={3}
                    value={aiPrompt}
                    onChange={(e) => setAiPrompt(e.target.value)}
                  />
                </div>
                
                <div>
                  <Label className="text-sm font-medium text-gray-700 mb-2 block">Content Tone</Label>
                  <Select>
                    <SelectTrigger>
                      <SelectValue placeholder="Select tone" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="professional">Professional</SelectItem>
                      <SelectItem value="casual">Casual</SelectItem>
                      <SelectItem value="enthusiastic">Enthusiastic</SelectItem>
                      <SelectItem value="humorous">Humorous</SelectItem>
                      <SelectItem value="inspirational">Inspirational</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <Button 
                className="w-full hover:scale-105 transition-transform duration-200"
                onClick={handleGenerateContent}
              >
                  <TrendingUp className="mr-2 h-4 w-4" />
                  Generate Content
                </Button>

                <div className="mt-6 p-4 bg-gray-50 rounded-lg">
                  <h4 className="font-medium text-gray-900 mb-2">AI Suggestions</h4>
                  <ul className="text-sm text-gray-600 space-y-1">
                    <li>• Include relevant hashtags for better reach</li>
                    <li>• Post during peak engagement hours</li>
                    <li>• Use emojis to increase interaction</li>
                    <li>• Ask questions to boost engagement</li>
                  </ul>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="posts">
          <Card>
            <CardHeader>
              <CardTitle>Recent Posts</CardTitle>
              <CardDescription>Your latest social media posts and their performance</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {recentPosts.map((post) => (
                  <div key={post.id} className="border rounded-lg p-4">
                    <div className="flex items-start justify-between">
                      <div className="flex items-start space-x-3">
                        <div className="w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center">
                          {post.platform === 'Instagram' && <Instagram className="h-4 w-4" />}
                          {post.platform === 'Twitter' && <Twitter className="h-4 w-4" />}
                          {post.platform === 'LinkedIn' && <Linkedin className="h-4 w-4" />}
                        </div>
                        <div className="flex-1">
                          <p className="text-gray-900 mb-2">{post.content}</p>
                          {post.image && (
                            <div className="w-full h-32 bg-gray-100 rounded-lg mb-3"></div>
                          )}
                          <div className="flex items-center space-x-4 text-sm text-gray-500">
                            <div className="flex items-center">
                              <Heart className="h-4 w-4 mr-1" />
                              {post.likes}
                            </div>
                            <div className="flex items-center">
                              <MessageCircle className="h-4 w-4 mr-1" />
                              {post.comments}
                            </div>
                            <div className="flex items-center">
                              <Share className="h-4 w-4 mr-1" />
                              {post.shares}
                            </div>
                            <span>• {post.postedAt}</span>
                          </div>
                        </div>
                      </div>
                      <Badge variant={post.performance === 'High' ? 'default' : 'secondary'}>
                        {post.performance}
                      </Badge>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="analytics">
          <Card>
            <CardHeader>
              <CardTitle>Social Media Analytics</CardTitle>
              <CardDescription>Detailed performance metrics across all platforms</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8">
                <TrendingUp className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">Analytics Dashboard</h3>
                <p className="text-gray-600">Advanced analytics coming soon</p>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="scheduler">
          <Card>
            <CardHeader>
              <CardTitle>Content Scheduler</CardTitle>
              <CardDescription>Plan and schedule your content in advance</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8">
                <Calendar className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">Content Calendar</h3>
                <p className="text-gray-600 mb-4">Schedule your posts for optimal engagement times</p>
                <Button>
                  <Plus className="mr-2 h-4 w-4" />
                  Create Schedule
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="trends">
          <div className="space-y-4 sm:space-y-6">
            {/* Trends Analysis Header */}
            <Card>
              <CardHeader className="pb-4">
                <CardTitle className="flex items-center text-lg sm:text-xl">
                  <Brain className="mr-2 h-5 w-5 text-purple-600" />
                  Social Media Trending Algorithm
                </CardTitle>
                <CardDescription className="text-sm">
                  AI-powered analysis of what's trending across social media platforms
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4 sm:space-y-6">
                {/* Analysis Configuration */}
                <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Timeframe</Label>
                    <Select value={selectedTimeframe} onValueChange={setSelectedTimeframe}>
                      <SelectTrigger className="text-sm">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="1h">Last Hour</SelectItem>
                        <SelectItem value="6h">Last 6 Hours</SelectItem>
                        <SelectItem value="24h">Last 24 Hours</SelectItem>
                        <SelectItem value="7d">Last 7 Days</SelectItem>
                        <SelectItem value="30d">Last 30 Days</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  
                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Category</Label>
                    <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                      <SelectTrigger className="text-sm">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="general">General</SelectItem>
                        <SelectItem value="tech">Technology</SelectItem>
                        <SelectItem value="fashion">Fashion</SelectItem>
                        <SelectItem value="food">Food & Beverage</SelectItem>
                        <SelectItem value="travel">Travel</SelectItem>
                        <SelectItem value="fitness">Fitness & Health</SelectItem>
                        <SelectItem value="entertainment">Entertainment</SelectItem>
                        <SelectItem value="business">Business</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  
                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Platforms</Label>
                    <div className="flex flex-wrap gap-1 sm:gap-2">
                      {platforms.map((platform) => {
                        const Icon = platform.icon
                        return (
                          <Button
                            key={platform.id}
                            variant={selectedTrendPlatforms.includes(platform.id) ? "default" : "outline"}
                            size="sm"
                            className={`text-xs ${
                              selectedTrendPlatforms.includes(platform.id) 
                                ? `bg-gradient-to-r ${platform.color} text-white` 
                                : ''
                            }`}
                            onClick={() => toggleTrendPlatform(platform.id)}
                          >
                            <Icon className="mr-1 h-3 w-3" />
                            <span className="hidden sm:inline">{platform.name}</span>
                            <span className="sm:hidden">{platform.name.slice(0, 2)}</span>
                          </Button>
                        )
                      })}
                    </div>
                  </div>
                </div>

                <Button 
                  onClick={analyzeTrends}
                  disabled={isAnalyzingTrends || selectedTrendPlatforms.length === 0}
                  className="w-full h-10 sm:h-12 text-sm sm:text-base"
                >
                  {isAnalyzingTrends ? (
                    <>
                      <div className="mr-2 h-4 w-4 animate-spin rounded-full border-2 border-gray-300 border-t-purple-600"></div>
                      Analyzing Trends...
                    </>
                  ) : (
                    <>
                      <Zap className="mr-2 h-4 w-4" />
                      Analyze Social Media Trends
                    </>
                  )}
                </Button>

                {/* Trends Error */}
                {trendsError && (
                  <div className="flex items-center p-3 bg-red-50 border border-red-200 rounded-lg">
                    <AlertCircle className="h-4 w-4 text-red-600 mr-2 flex-shrink-0" />
                    <span className="text-red-700 text-sm">{trendsError}</span>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Trends Results */}
            {trendsData && (
              <div className="grid grid-cols-1 xl:grid-cols-2 gap-4 sm:gap-6">
                {/* Top Trends */}
                <Card>
                  <CardHeader className="pb-4">
                    <CardTitle className="flex items-center text-lg sm:text-xl">
                      <TrendingUp className="mr-2 h-5 w-5 text-green-600" />
                      Top Trending Topics
                    </CardTitle>
                    <CardDescription className="text-sm">
                      Currently trending across selected platforms
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-3 sm:space-y-4 max-h-80 sm:max-h-96 overflow-y-auto">
                      {trendsData.trends?.map((trend: any, index: number) => (
                        <div key={index} className="border rounded-lg p-3 sm:p-4 hover:shadow-md transition-shadow">
                          <div className="flex items-start justify-between">
                            <div className="flex-1 min-w-0">
                              <div className="flex items-center space-x-2 mb-2">
                                <h5 className="font-medium text-gray-900 text-sm sm:text-base truncate">{trend.topic}</h5>
                                <Badge variant="secondary" className="text-xs">{trend.hashtag}</Badge>
                              </div>
                              <p className="text-xs sm:text-sm text-gray-600 mb-3 line-clamp-2">{trend.description}</p>
                              
                              <div className="flex flex-wrap items-center gap-2 sm:gap-4 text-xs">
                                <Badge className={`${getEngagementColor(trend.engagement)} text-xs`}>
                                  {trend.engagement} engagement
                                </Badge>
                                <span className="text-green-600 font-medium">{trend.growth}</span>
                                <span className={getSentimentColor(trend.sentiment)}>
                                  {trend.sentiment}
                                </span>
                              </div>

                              <div className="flex items-center space-x-2 mt-3">
                                {trend.platforms?.map((platform: string, pIndex: number) => {
                                  const platformData = platforms.find(p => p.id === platform)
                                  if (!platformData) return null
                                  const Icon = platformData.icon
                                  return (
                                    <div key={pIndex} className={`w-5 h-5 sm:w-6 sm:h-6 bg-gradient-to-r ${platformData.color} rounded flex items-center justify-center`}>
                                      <Icon className="h-2.5 w-2.5 sm:h-3 sm:w-3 text-white" />
                                    </div>
                                  )
                                })}
                              </div>

                              <p className="text-xs text-blue-600 mt-2 line-clamp-2">
                                <strong>Opportunity:</strong> {trend.opportunity}
                              </p>
                            </div>
                            
                            <Button
                              size="sm"
                              onClick={() => generateContentFromTrend(trend)}
                              disabled={isGeneratingContent}
                              className="ml-2 flex-shrink-0 h-8 w-8 sm:h-10 sm:w-10 p-0"
                            >
                              {isGeneratingContent ? (
                                <div className="h-3 w-3 sm:h-4 sm:w-4 animate-spin rounded-full border-2 border-gray-300 border-t-white"></div>
                              ) : (
                                <Sparkles className="h-3 w-3 sm:h-4 sm:w-4" />
                              )}
                            </Button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>

                {/* Insights & Recommendations */}
                <div className="space-y-4 sm:space-y-6">
                  {/* Key Insights */}
                  <Card>
                    <CardHeader className="pb-4">
                      <CardTitle className="flex items-center text-lg sm:text-xl">
                        <BarChart3 className="mr-2 h-5 w-5 text-blue-600" />
                        Key Insights
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-3 sm:space-y-4">
                        <div>
                          <h5 className="font-medium text-gray-900 mb-2 text-sm sm:text-base">Best Posting Times</h5>
                          <div className="flex flex-wrap gap-1 sm:gap-2">
                            {trendsData.insights?.bestPostingTimes?.map((time: string, index: number) => (
                              <Badge key={index} variant="outline" className="text-xs">
                                <Clock className="mr-1 h-3 w-3" />
                                {time}
                              </Badge>
                            ))}
                          </div>
                        </div>

                        <div>
                          <h5 className="font-medium text-gray-900 mb-2 text-sm sm:text-base">Top Content Formats</h5>
                          <div className="flex flex-wrap gap-1 sm:gap-2">
                            {trendsData.insights?.topFormats?.map((format: string, index: number) => (
                              <Badge key={index} className="bg-purple-100 text-purple-800 text-xs">
                                {format}
                              </Badge>
                            ))}
                          </div>
                        </div>

                        <div>
                          <h5 className="font-medium text-gray-900 mb-2 text-sm sm:text-base">Emerging Trends</h5>
                          <div className="flex flex-wrap gap-1 sm:gap-2">
                            {trendsData.insights?.emergingTrends?.map((trend: string, index: number) => (
                              <Badge key={index} className="bg-green-100 text-green-800 text-xs">
                                <TrendingUp className="mr-1 h-3 w-3" />
                                {trend}
                              </Badge>
                            ))}
                          </div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>

                  {/* AI Recommendations */}
                  <Card>
                    <CardHeader className="pb-4">
                      <CardTitle className="flex items-center text-lg sm:text-xl">
                        <Target className="mr-2 h-5 w-5 text-orange-600" />
                        AI Recommendations
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-2 sm:space-y-3">
                        {trendsData.recommendations?.map((recommendation: string, index: number) => (
                          <div key={index} className="flex items-start space-x-2 sm:space-x-3">
                            <Check className="h-3 w-3 sm:h-4 sm:w-4 text-green-600 mt-0.5 flex-shrink-0" />
                            <p className="text-xs sm:text-sm text-gray-700">{recommendation}</p>
                          </div>
                        ))}
                      </div>
                    </CardContent>
                  </Card>
                </div>
              </div>
            )}

            {/* Generated Content */}
            {generatedContent && (
              <Card>
                <CardHeader className="pb-4">
                  <CardTitle className="flex items-center text-lg sm:text-xl">
                    <Sparkles className="mr-2 h-5 w-5 text-yellow-600" />
                    AI-Generated Content
                  </CardTitle>
                  <CardDescription className="text-sm">
                    Content created based on trending topics
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-3 sm:space-y-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Headline</Label>
                    <p className="font-medium text-gray-900 text-sm sm:text-base">{generatedContent.headline}</p>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Content</Label>
                    <Textarea
                      value={generatedContent.content}
                      onChange={(e) => setContent(e.target.value)}
                      rows={4}
                      className="w-full text-sm"
                    />
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Hashtags</Label>
                    <div className="flex flex-wrap gap-1 sm:gap-2">
                      {generatedContent.hashtags?.map((hashtag: string, index: number) => (
                        <Badge key={index} variant="secondary" className="text-xs">{hashtag}</Badge>
                      ))}
                    </div>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Call to Action</Label>
                    <p className="text-gray-700 text-sm sm:text-base">{generatedContent.callToAction}</p>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Visual Suggestions</Label>
                    <div className="space-y-1 sm:space-y-2">
                      {generatedContent.visualSuggestions?.map((suggestion: string, index: number) => (
                        <div key={index} className="flex items-center space-x-2 text-xs sm:text-sm text-gray-600">
                          <ImageIcon className="h-3 w-3 sm:h-4 sm:w-4 flex-shrink-0" />
                          <span className="line-clamp-1">{suggestion}</span>
                        </div>
                      ))}
                    </div>
                  </div>

                  <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                    <div>
                      <Label className="text-sm font-medium text-gray-700 mb-2 block">Best Posting Time</Label>
                      <p className="text-xs sm:text-sm text-gray-600">{generatedContent.bestPostingTime}</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-700 mb-2 block">Platform</Label>
                      <p className="text-xs sm:text-sm text-gray-600 capitalize">{generatedContent.platform}</p>
                    </div>
                  </div>

                  {/* Content Error */}
                  {contentError && (
                    <div className="flex items-center p-3 bg-red-50 border border-red-200 rounded-lg">
                      <AlertCircle className="h-4 w-4 text-red-600 mr-2 flex-shrink-0" />
                      <span className="text-red-700 text-sm">{contentError}</span>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}
          </div>
        </TabsContent>
      </Tabs>
    </div>
  )
}