'use client'

import { useState } from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Textarea } from '@/components/ui/textarea'
import { Badge } from '@/components/ui/badge'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Label } from '@/components/ui/label'
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog'
import { toast } from 'sonner'
import { 
  Search, 
  Globe, 
  TrendingUp, 
  MapPin, 
  Target,
  Brain,
  Plus,
  ArrowUpRight,
  ArrowDownRight,
  BarChart3,
  FileText,
  Eye,
  Link,
  X,
  Check,
  AlertCircle,
  ExternalLink,
  RefreshCw,
  Download,
  Upload
} from 'lucide-react'

export function SeoTab() {
  const [selectedDomain, setSelectedDomain] = useState('')
  const [targetKeywords, setTargetKeywords] = useState<string[]>([])
  const [targetCountry, setTargetCountry] = useState('US')
  const [targetState, setTargetState] = useState('')
  const [targetCity, setTargetCity] = useState('')
  const [keywordInput, setKeywordInput] = useState('')
  const [isDomainModalOpen, setIsDomainModalOpen] = useState(false)
  const [newDomain, setNewDomain] = useState('')
  const [domainDescription, setDomainDescription] = useState('')
  const [domainError, setDomainError] = useState('')
  const [domainSuccess, setDomainSuccess] = useState('')
  const [isAnalyzing, setIsAnalyzing] = useState(false)
  const [analysisResults, setAnalysisResults] = useState<any[]>([])
  const [analysisError, setAnalysisError] = useState('')
  
  // SEO Utilities state
  const [backlinkUrl, setBacklinkUrl] = useState('')
  const [backlinkResults, setBacklinkResults] = useState<any[]>([])
  const [isBacklinkChecking, setIsBacklinkChecking] = useState(false)
  const [backlinkModalOpen, setBacklinkModalOpen] = useState(false)
  
  const [contentOptimizerUrl, setContentOptimizerUrl] = useState('')
  const [contentOptimizerText, setContentOptimizerText] = useState('')
  const [contentOptimizationResults, setContentOptimizationResults] = useState<any>(null)
  const [isContentOptimizing, setIsContentOptimizing] = useState(false)
  const [contentOptimizerModalOpen, setContentOptimizerModalOpen] = useState(false)
  
  const [siteAuditUrl, setSiteAuditUrl] = useState('')
  const [siteAuditResults, setSiteAuditResults] = useState<any>(null)
  const [isSiteAuditing, setIsSiteAuditing] = useState(false)
  const [siteAuditModalOpen, setSiteAuditModalOpen] = useState(false)
  
  const [competitorUrl, setCompetitorUrl] = useState('')
  const [competitorResults, setCompetitorResults] = useState<any>(null)
  const [isCompetitorAnalyzing, setIsCompetitorAnalyzing] = useState(false)
  const [competitorModalOpen, setCompetitorModalOpen] = useState(false)

  const domains = [
    { id: '1', name: 'mywebsite.com', url: 'https://mywebsite.com' },
    { id: '2', name: 'blog.mywebsite.com', url: 'https://blog.mywebsite.com' }
  ]

  const keywordData = [
    {
      keyword: 'social media marketing',
      volume: 45000,
      difficulty: 65,
      position: 12,
      change: '+3',
      intent: 'commercial'
    },
    {
      keyword: 'ai content generator',
      volume: 22000,
      difficulty: 58,
      position: 8,
      change: '+5',
      intent: 'transactional'
    },
    {
      keyword: 'instagram growth tool',
      volume: 18000,
      difficulty: 72,
      position: 15,
      change: '-2',
      intent: 'commercial'
    },
    {
      keyword: 'best seo practices',
      volume: 35000,
      difficulty: 68,
      position: 22,
      change: '+8',
      intent: 'informational'
    }
  ]

  const rankingData = [
    {
      keyword: 'social media marketing',
      position: 12,
      url: '/blog/social-media-marketing',
      country: 'United States',
      state: 'California',
      city: 'Los Angeles',
      change: '+3'
    },
    {
      keyword: 'ai content generator',
      position: 8,
      url: '/tools/ai-content-generator',
      country: 'United States',
      state: 'New York',
      city: 'New York',
      change: '+5'
    },
    {
      keyword: 'instagram growth',
      position: 15,
      url: '/services/instagram-growth',
      country: 'United Kingdom',
      state: '',
      city: 'London',
      change: '-2'
    }
  ]

  const countries = [
    { code: 'US', name: 'United States' },
    { code: 'GB', name: 'United Kingdom' },
    { code: 'CA', name: 'Canada' },
    { code: 'AU', name: 'Australia' },
    { code: 'DE', name: 'Germany' }
  ]

  const addKeyword = () => {
    if (keywordInput.trim()) {
      setTargetKeywords(prev => [...prev, keywordInput.trim()])
      setKeywordInput('')
    }
  }

  const removeKeyword = (index: number) => {
    setTargetKeywords(prev => prev.filter((_, i) => i !== index))
  }

  const validateDomain = (domain: string) => {
    const domainRegex = /^(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?\.)+[a-zA-Z]{2,}$/
    return domainRegex.test(domain)
  }

  const handleAddDomain = () => {
    setDomainError('')
    setDomainSuccess('')
    
    if (!newDomain.trim()) {
      setDomainError('Please enter a domain name')
      return
    }
    
    if (!validateDomain(newDomain.trim())) {
      setDomainError('Please enter a valid domain name (e.g., example.com)')
      return
    }
    
    // Simulate adding domain
    setTimeout(() => {
      setDomainSuccess(`Domain "${newDomain.trim()}" has been successfully added!`)
      setNewDomain('')
      setDomainDescription('')
      
      // Close modal after success
      setTimeout(() => {
        setIsDomainModalOpen(false)
        setDomainSuccess('')
      }, 2000)
    }, 1000)
  }

  const resetDomainForm = () => {
    setNewDomain('')
    setDomainDescription('')
    setDomainError('')
    setDomainSuccess('')
  }

  const getDifficultyColor = (difficulty: number | string) => {
    if (typeof difficulty === 'string') {
      switch (difficulty.toLowerCase()) {
        case 'easy': return 'text-green-600'
        case 'medium': return 'text-yellow-600'
        case 'hard': return 'text-red-600'
        default: return 'text-gray-600'
      }
    }
    if (difficulty < 30) return 'text-green-600'
    if (difficulty < 60) return 'text-yellow-600'
    return 'text-red-600'
  }

  const getIntentColor = (intent: string) => {
    switch (intent) {
      case 'commercial': return 'bg-blue-100 text-blue-800'
      case 'transactional': return 'bg-green-100 text-green-800'
      case 'informational': return 'bg-purple-100 text-purple-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const analyzeKeywords = async () => {
    if (targetKeywords.length === 0 && !keywordInput.trim()) {
      setAnalysisError('Please add at least one keyword to analyze')
      return
    }

    setIsAnalyzing(true)
    setAnalysisError('')
    setAnalysisResults([])

    try {
      const keywordsToAnalyze = targetKeywords.length > 0 ? targetKeywords.join(', ') : keywordInput.trim()
      
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'keywords',
          prompt: keywordsToAnalyze,
          options: {
            domain: selectedDomain,
            targetCountry: targetCountry
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to analyze keywords')
      }

      const data = await response.json()
      
      if (data.keywords && Array.isArray(data.keywords)) {
        setAnalysisResults(data.keywords)
      } else {
        setAnalysisError('No keyword data received')
      }
    } catch (error) {
      console.error('Error analyzing keywords:', error)
      setAnalysisError('Failed to analyze keywords. Please try again.')
    } finally {
      setIsAnalyzing(false)
    }
  }

  // SEO Utility Handlers
  const handleBacklinkChecker = () => {
    setBacklinkModalOpen(true)
  }

  const checkBacklinks = async () => {
    if (!backlinkUrl.trim()) {
      toast.error('Please enter a URL to check backlinks')
      return
    }

    setIsBacklinkChecking(true)
    toast.loading('Checking backlinks...')

    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'trends',
          prompt: `Analyze backlinks for ${backlinkUrl}`,
          options: {
            analysis: 'backlinks'
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to check backlinks')
      }

      const data = await response.json()
      
      // Mock backlink results for now
      const mockBacklinks = [
        {
          source: 'techblog.com',
          url: 'https://techblog.com/article-about-seo',
          authority: 75,
          relevance: 'high',
          status: 'active'
        },
        {
          source: 'marketingguide.org',
          url: 'https://marketingguide.org/seo-tips',
          authority: 62,
          relevance: 'medium',
          status: 'active'
        },
        {
          source: 'digitalnews.net',
          url: 'https://digitalnews.net/tech-updates',
          authority: 58,
          relevance: 'low',
          status: 'broken'
        }
      ]

      setBacklinkResults(mockBacklinks)
      toast.success(`Found ${mockBacklinks.length} backlinks`)
    } catch (error) {
      console.error('Error checking backlinks:', error)
      toast.error('Failed to check backlinks')
    } finally {
      setIsBacklinkChecking(false)
    }
  }

  const handleContentOptimizer = () => {
    setContentOptimizerModalOpen(true)
  }

  const optimizeContent = async () => {
    if (!contentOptimizerText.trim() && !contentOptimizerUrl.trim()) {
      toast.error('Please enter content or URL to optimize')
      return
    }

    setIsContentOptimizing(true)
    toast.loading('Optimizing content...')

    try {
      const contentToAnalyze = contentOptimizerText || contentOptimizerUrl
      
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'optimize',
          prompt: contentToAnalyze,
          options: {
            analysis: 'seo_optimization'
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to optimize content')
      }

      const data = await response.json()
      
      // Mock optimization results
      const mockResults = {
        score: 78,
        suggestions: [
          'Add more relevant keywords to your title',
          'Improve meta description length (current: 145, ideal: 155-160)',
          'Add internal links to related content',
          'Optimize image alt texts',
          'Improve readability score'
        ],
        keywordDensity: {
          'seo tools': 2.1,
          'optimization': 1.8,
          'content': 3.2
        },
        readability: {
          score: 72,
          grade: '8th grade',
          suggestions: ['Use shorter sentences', 'Break up long paragraphs']
        }
      }

      setContentOptimizationResults(mockResults)
      toast.success('Content optimization complete!')
    } catch (error) {
      console.error('Error optimizing content:', error)
      toast.error('Failed to optimize content')
    } finally {
      setIsContentOptimizing(false)
    }
  }

  const handleSiteAuditor = () => {
    setSiteAuditModalOpen(true)
  }

  const auditSite = async () => {
    if (!siteAuditUrl.trim()) {
      toast.error('Please enter a URL to audit')
      return
    }

    setIsSiteAuditing(true)
    toast.loading('Auditing website...')

    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'trends',
          prompt: `Perform comprehensive SEO audit for ${siteAuditUrl}`,
          options: {
            analysis: 'site_audit'
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to audit site')
      }

      const data = await response.json()
      
      // Mock site audit results
      const mockResults = {
        overallScore: 82,
        categories: {
          technical: {
            score: 88,
            issues: ['Missing sitemap.xml', 'Slow page load speed'],
            fixes: ['Create sitemap.xml', 'Optimize images and caching']
          },
          content: {
            score: 75,
            issues: ['Thin content on some pages', 'Missing meta descriptions'],
            fixes: ['Add more detailed content', 'Write unique meta descriptions']
          },
          performance: {
            score: 79,
            issues: ['Large image files', 'Unoptimized CSS'],
            fixes: ['Compress images', 'Minify CSS files']
          }
        },
        recommendations: [
          'Implement structured data markup',
          'Improve mobile responsiveness',
          'Add SSL certificate',
          'Optimize for Core Web Vitals'
        ]
      }

      setSiteAuditResults(mockResults)
      toast.success('Site audit complete!')
    } catch (error) {
      console.error('Error auditing site:', error)
      toast.error('Failed to audit site')
    } finally {
      setIsSiteAuditing(false)
    }
  }

  const handleCompetitorAnalysis = () => {
    setCompetitorModalOpen(true)
  }

  const analyzeCompetitor = async () => {
    if (!competitorUrl.trim()) {
      toast.error('Please enter a competitor URL to analyze')
      return
    }

    setIsCompetitorAnalyzing(true)
    toast.loading('Analyzing competitor...')

    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'trends',
          prompt: `Analyze competitor SEO strategy for ${competitorUrl}`,
          options: {
            analysis: 'competitor_analysis'
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to analyze competitor')
      }

      const data = await response.json()
      
      // Mock competitor analysis results
      const mockResults = {
        competitor: competitorUrl,
        overview: {
          domainAuthority: 68,
          organicKeywords: 1250,
          monthlyTraffic: '45.2K',
          backlinks: 892
        },
        topKeywords: [
          { keyword: 'seo optimization', position: 3, volume: 22000 },
          { keyword: 'content marketing', position: 7, volume: 18000 },
          { keyword: 'digital marketing tools', position: 5, volume: 15000 }
        ],
        strengths: [
          'Strong content marketing strategy',
          'High-quality backlink profile',
          'Good mobile optimization',
          'Fast page load speeds'
        ],
        weaknesses: [
          'Limited social media presence',
          'Poor local SEO optimization',
          'Outdated technical SEO',
          'Weak internal linking'
        ],
        opportunities: [
          'Target long-tail keywords they miss',
          'Create better content for their top keywords',
          'Improve local SEO presence',
          'Build more diverse backlink profile'
        ]
      }

      setCompetitorResults(mockResults)
      toast.success('Competitor analysis complete!')
    } catch (error) {
      console.error('Error analyzing competitor:', error)
      toast.error('Failed to analyze competitor')
    } finally {
      setIsCompetitorAnalyzing(false)
    }
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">SEO Tools</h1>
          <p className="text-gray-600 mt-2">Optimize your website for search engines</p>
        </div>
        <Dialog open={isDomainModalOpen} onOpenChange={setIsDomainModalOpen}>
          <DialogTrigger asChild>
            <Button className="hover:scale-105 transition-transform cursor-pointer">
              <Plus className="mr-2 h-4 w-4" />
              Add Domain
            </Button>
          </DialogTrigger>
          <DialogContent className="sm:max-w-md">
            <DialogHeader>
              <DialogTitle className="flex items-center">
                <Globe className="mr-2 h-5 w-5 text-blue-600" />
                Add New Domain
              </DialogTitle>
              <DialogDescription>
                Add a new domain to track SEO performance and rankings
              </DialogDescription>
            </DialogHeader>
            
            <div className="space-y-4 mt-4">
              <div>
                <Label htmlFor="domain-name" className="text-sm font-medium text-gray-700 mb-2 block">
                  Domain Name *
                </Label>
                <Input
                  id="domain-name"
                  placeholder="example.com"
                  value={newDomain}
                  onChange={(e) => setNewDomain(e.target.value)}
                  className={domainError ? 'border-red-300' : ''}
                />
                {domainError && (
                  <div className="flex items-center mt-2 text-red-600 text-sm">
                    <AlertCircle className="h-4 w-4 mr-1" />
                    {domainError}
                  </div>
                )}
              </div>

              <div>
                <Label htmlFor="domain-description" className="text-sm font-medium text-gray-700 mb-2 block">
                  Description (Optional)
                </Label>
                <Textarea
                  id="domain-description"
                  placeholder="Brief description of the website or project..."
                  value={domainDescription}
                  onChange={(e) => setDomainDescription(e.target.value)}
                  rows={3}
                />
              </div>

              {domainSuccess && (
                <div className="flex items-center p-3 bg-green-50 border border-green-200 rounded-lg">
                  <Check className="h-4 w-4 text-green-600 mr-2" />
                  <span className="text-green-700 text-sm">{domainSuccess}</span>
                </div>
              )}

              <div className="flex justify-end space-x-3 pt-4">
                <Button
                  variant="outline"
                  onClick={() => {
                    setIsDomainModalOpen(false)
                    resetDomainForm()
                  }}
                  disabled={domainSuccess !== ''}
                >
                  Cancel
                </Button>
                <Button
                  onClick={handleAddDomain}
                  disabled={domainSuccess !== ''}
                  className="hover:scale-105 transition-transform"
                >
                  {domainSuccess ? 'Adding...' : 'Add Domain'}
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      {/* Domain Overview */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Total Keywords</p>
                <p className="text-2xl font-bold text-gray-900 mt-1">47</p>
              </div>
              <div className="p-3 bg-blue-50 rounded-lg">
                <Target className="h-6 w-6 text-blue-600" />
              </div>
            </div>
            <div className="flex items-center mt-4">
              <ArrowUpRight className="h-4 w-4 text-green-600 mr-1" />
              <span className="text-sm font-medium text-green-600">+12.5%</span>
              <span className="text-sm text-gray-500 ml-2">vs last month</span>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Avg. Position</p>
                <p className="text-2xl font-bold text-gray-900 mt-1">18.3</p>
              </div>
              <div className="p-3 bg-green-50 rounded-lg">
                <BarChart3 className="h-6 w-6 text-green-600" />
              </div>
            </div>
            <div className="flex items-center mt-4">
              <ArrowUpRight className="h-4 w-4 text-green-600 mr-1" />
              <span className="text-sm font-medium text-green-600">+5.2</span>
              <span className="text-sm text-gray-500 ml-2">positions improved</span>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Organic Traffic</p>
                <p className="text-2xl font-bold text-gray-900 mt-1">12.4K</p>
              </div>
              <div className="p-3 bg-purple-50 rounded-lg">
                <Eye className="h-6 w-6 text-purple-600" />
              </div>
            </div>
            <div className="flex items-center mt-4">
              <ArrowUpRight className="h-4 w-4 text-green-600 mr-1" />
              <span className="text-sm font-medium text-green-600">+23.1%</span>
              <span className="text-sm text-gray-500 ml-2">vs last month</span>
            </div>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="keywords" className="space-y-6">
        <TabsList>
          <TabsTrigger value="keywords">Keyword Research</TabsTrigger>
          <TabsTrigger value="rankings">Ranking Tracker</TabsTrigger>
          <TabsTrigger value="content">AI Content Generator</TabsTrigger>
          <TabsTrigger value="geo">Geographic Rankings</TabsTrigger>
        </TabsList>

        <TabsContent value="keywords">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Keyword Research Tool */}
            <Card className="lg:col-span-2">
              <CardHeader>
                <CardTitle>Keyword Research</CardTitle>
                <CardDescription>Find and analyze target keywords</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="flex space-x-3">
                  <div className="flex-1">
                    <Input
                      placeholder="Enter keywords to research..."
                      value={keywordInput}
                      onChange={(e) => setKeywordInput(e.target.value)}
                      onKeyPress={(e) => e.key === 'Enter' && addKeyword()}
                    />
                  </div>
                  <Button onClick={addKeyword}>
                    <Plus className="h-4 w-4" />
                  </Button>
                </div>

                {targetKeywords.length > 0 && (
                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Target Keywords</Label>
                    <div className="flex flex-wrap gap-2">
                      {targetKeywords.map((keyword, index) => (
                        <Badge key={index} variant="secondary" className="cursor-pointer">
                          {keyword}
                          <button
                            onClick={() => removeKeyword(index)}
                            className="ml-2 text-gray-500 hover:text-gray-700"
                          >
                            ×
                          </button>
                        </Badge>
                      ))}
                    </div>
                  </div>
                )}

                <Button 
                  className="w-full" 
                  onClick={analyzeKeywords}
                  disabled={isAnalyzing || (targetKeywords.length === 0 && !keywordInput.trim())}
                >
                  {isAnalyzing ? (
                    <>
                      <div className="mr-2 h-4 w-4 animate-spin rounded-full border-2 border-gray-300 border-t-blue-600"></div>
                      Analyzing...
                    </>
                  ) : (
                    <>
                      <Search className="mr-2 h-4 w-4" />
                      Analyze Keywords
                    </>
                  )}
                </Button>

                {/* Analysis Error */}
                {analysisError && (
                  <div className="flex items-center p-3 bg-red-50 border border-red-200 rounded-lg">
                    <AlertCircle className="h-4 w-4 text-red-600 mr-2" />
                    <span className="text-red-700 text-sm">{analysisError}</span>
                  </div>
                )}

                {/* Keyword Results */}
                <div className="space-y-4">
                  <h4 className="font-medium text-gray-900">
                    {analysisResults.length > 0 ? 'AI Keyword Analysis Results' : 'Sample Keyword Analysis'}
                  </h4>
                  {analysisResults.length > 0 ? (
                    analysisResults.map((keyword, index) => (
                      <div key={index} className="border rounded-lg p-4">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center space-x-3">
                              <h5 className="font-medium text-gray-900">{keyword.keyword}</h5>
                              <Badge className={getIntentColor(keyword.intent)}>
                                {keyword.intent}
                              </Badge>
                            </div>
                            <div className="flex items-center space-x-6 mt-2 text-sm text-gray-600">
                              <span>Volume: {keyword.volume}</span>
                              <span className={getDifficultyColor(keyword.difficulty)}>
                                Difficulty: {keyword.difficulty}
                              </span>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))
                  ) : (
                    keywordData.map((keyword, index) => (
                      <div key={index} className="border rounded-lg p-4">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center space-x-3">
                              <h5 className="font-medium text-gray-900">{keyword.keyword}</h5>
                              <Badge className={getIntentColor(keyword.intent)}>
                                {keyword.intent}
                              </Badge>
                            </div>
                            <div className="flex items-center space-x-6 mt-2 text-sm text-gray-600">
                              <span>Volume: {keyword.volume.toLocaleString()}</span>
                              <span className={getDifficultyColor(keyword.difficulty)}>
                                Difficulty: {keyword.difficulty}
                              </span>
                              <span>Position: #{keyword.position}</span>
                            </div>
                          </div>
                          <div className="flex items-center">
                            {keyword.change.startsWith('+') ? (
                              <ArrowUpRight className="h-4 w-4 text-green-600 mr-1" />
                            ) : (
                              <ArrowDownRight className="h-4 w-4 text-red-600 mr-1" />
                            )}
                            <span className={`text-sm font-medium ${
                              keyword.change.startsWith('+') ? 'text-green-600' : 'text-red-600'
                            }`}>
                              {keyword.change}
                            </span>
                          </div>
                        </div>
                      </div>
                    ))
                  )}
                </div>
              </CardContent>
            </Card>

            {/* SEO Tools */}
            <Card>
              <CardHeader>
                <CardTitle>SEO Tools</CardTitle>
                <CardDescription>Quick SEO utilities</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <Button 
                  variant="outline" 
                  className="w-full justify-start hover:bg-blue-50 hover:border-blue-200 transition-colors cursor-pointer"
                  onClick={handleBacklinkChecker}
                >
                  <Link className="mr-2 h-4 w-4" />
                  Backlink Checker
                </Button>
                <Button 
                  variant="outline" 
                  className="w-full justify-start hover:bg-blue-50 hover:border-blue-200 transition-colors cursor-pointer"
                  onClick={handleContentOptimizer}
                >
                  <FileText className="mr-2 h-4 w-4" />
                  Content Optimizer
                </Button>
                <Button 
                  variant="outline" 
                  className="w-full justify-start hover:bg-blue-50 hover:border-blue-200 transition-colors cursor-pointer"
                  onClick={handleSiteAuditor}
                >
                  <Globe className="mr-2 h-4 w-4" />
                  Site Auditor
                </Button>
                <Button 
                  variant="outline" 
                  className="w-full justify-start hover:bg-blue-50 hover:border-blue-200 transition-colors cursor-pointer"
                  onClick={handleCompetitorAnalysis}
                >
                  <Search className="mr-2 h-4 w-4" />
                  Competitor Analysis
                </Button>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="rankings">
          <Card>
            <CardHeader>
              <CardTitle>Ranking Tracker</CardTitle>
              <CardDescription>Monitor your search engine rankings</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {rankingData.map((ranking, index) => (
                  <div key={index} className="border rounded-lg p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex-1">
                        <div className="flex items-center space-x-3">
                          <h5 className="font-medium text-gray-900">{ranking.keyword}</h5>
                          <Badge variant="outline">#{ranking.position}</Badge>
                        </div>
                        <div className="flex items-center space-x-6 mt-2 text-sm text-gray-600">
                          <span>{ranking.url}</span>
                          <div className="flex items-center">
                            <MapPin className="h-3 w-3 mr-1" />
                            {ranking.city}, {ranking.state || ranking.country}
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center">
                        {ranking.change.startsWith('+') ? (
                          <ArrowUpRight className="h-4 w-4 text-green-600 mr-1" />
                        ) : (
                          <ArrowDownRight className="h-4 w-4 text-red-600 mr-1" />
                        )}
                        <span className={`text-sm font-medium ${
                          ranking.change.startsWith('+') ? 'text-green-600' : 'text-red-600'
                        }`}>
                          {ranking.change}
                        </span>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="content">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* AI Content Generator */}
            <Card>
              <CardHeader>
                <CardTitle>AI Content Generator</CardTitle>
                <CardDescription>Create SEO-optimized content with AI</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="domain" className="text-sm font-medium text-gray-700 mb-2 block">
                    Target Domain
                  </Label>
                  <Select value={selectedDomain} onValueChange={setSelectedDomain}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select domain" />
                    </SelectTrigger>
                    <SelectContent>
                      {domains.map((domain) => (
                        <SelectItem key={domain.id} value={domain.id}>
                          {domain.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="primary-keyword" className="text-sm font-medium text-gray-700 mb-2 block">
                    Primary Keyword
                  </Label>
                  <Input
                    id="primary-keyword"
                    placeholder="Enter primary keyword..."
                  />
                </div>

                <div>
                  <Label htmlFor="content-type" className="text-sm font-medium text-gray-700 mb-2 block">
                    Content Type
                  </Label>
                  <Select>
                    <SelectTrigger>
                      <SelectValue placeholder="Select content type" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="blog-post">Blog Post</SelectItem>
                      <SelectItem value="landing-page">Landing Page</SelectItem>
                      <SelectItem value="product-description">Product Description</SelectItem>
                      <SelectItem value="category-page">Category Page</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="tone" className="text-sm font-medium text-gray-700 mb-2 block">
                    Content Tone
                  </Label>
                  <Select>
                    <SelectTrigger>
                      <SelectValue placeholder="Select tone" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="professional">Professional</SelectItem>
                      <SelectItem value="casual">Casual</SelectItem>
                      <SelectItem value="persuasive">Persuasive</SelectItem>
                      <SelectItem value="informative">Informative</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <Button className="w-full">
                  <Brain className="mr-2 h-4 w-4" />
                  Generate Content
                </Button>
              </CardContent>
            </Card>

            {/* Generated Content Preview */}
            <Card>
              <CardHeader>
                <CardTitle>Generated Content</CardTitle>
                <CardDescription>Preview and edit your AI-generated content</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="text-center py-8">
                  <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">Content Preview</h3>
                  <p className="text-gray-600">Generated content will appear here</p>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="geo">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Geographic Targeting */}
            <Card>
              <CardHeader>
                <CardTitle>Geographic Targeting</CardTitle>
                <CardDescription>Track rankings in specific locations</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="domain-geo" className="text-sm font-medium text-gray-700 mb-2 block">
                    Domain
                  </Label>
                  <Select>
                    <SelectTrigger>
                      <SelectValue placeholder="Select domain" />
                    </SelectTrigger>
                    <SelectContent>
                      {domains.map((domain) => (
                        <SelectItem key={domain.id} value={domain.id}>
                          {domain.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="country" className="text-sm font-medium text-gray-700 mb-2 block">
                    Country
                  </Label>
                  <Select value={targetCountry} onValueChange={setTargetCountry}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select country" />
                    </SelectTrigger>
                    <SelectContent>
                      {countries.map((country) => (
                        <SelectItem key={country.code} value={country.code}>
                          {country.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="state" className="text-sm font-medium text-gray-700 mb-2 block">
                    State/Province
                  </Label>
                  <Input
                    id="state"
                    placeholder="Enter state or province..."
                    value={targetState}
                    onChange={(e) => setTargetState(e.target.value)}
                  />
                </div>

                <div>
                  <Label htmlFor="city" className="text-sm font-medium text-gray-700 mb-2 block">
                    City
                  </Label>
                  <Input
                    id="city"
                    placeholder="Enter city..."
                    value={targetCity}
                    onChange={(e) => setTargetCity(e.target.value)}
                  />
                </div>

                <Button className="w-full">
                  <MapPin className="mr-2 h-4 w-4" />
                  Track Geographic Rankings
                </Button>
              </CardContent>
            </Card>

            {/* Geographic Rankings Map */}
            <Card>
              <CardHeader>
                <CardTitle>Geographic Rankings</CardTitle>
                <CardDescription>Visual representation of your rankings</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="text-center py-8">
                  <Globe className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">Rankings Map</h3>
                  <p className="text-gray-600 mb-4">Interactive map showing geographic rankings</p>
                  <Button variant="outline">
                    <Eye className="mr-2 h-4 w-4" />
                    View Map
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>
      </Tabs>

      {/* Backlink Checker Modal */}
      <Dialog open={backlinkModalOpen} onOpenChange={setBacklinkModalOpen}>
        <DialogContent className="sm:max-w-2xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center">
              <Link className="mr-2 h-5 w-5 text-blue-600" />
              Backlink Checker
            </DialogTitle>
            <DialogDescription>
              Analyze backlinks pointing to your website
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 mt-4">
            <div>
              <Label htmlFor="backlink-url" className="text-sm font-medium text-gray-700 mb-2 block">
                Website URL *
              </Label>
              <Input
                id="backlink-url"
                placeholder="https://example.com"
                value={backlinkUrl}
                onChange={(e) => setBacklinkUrl(e.target.value)}
              />
            </div>

            <Button 
              onClick={checkBacklinks}
              disabled={isBacklinkChecking || !backlinkUrl.trim()}
              className="w-full"
            >
              {isBacklinkChecking ? (
                <>
                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                  Checking Backlinks...
                </>
              ) : (
                <>
                  <Search className="mr-2 h-4 w-4" />
                  Check Backlinks
                </>
              )}
            </Button>

            {backlinkResults.length > 0 && (
              <div className="space-y-3">
                <h4 className="font-medium text-gray-900">Backlink Results</h4>
                <div className="space-y-2">
                  {backlinkResults.map((backlink, index) => (
                    <div key={index} className="p-3 border rounded-lg">
                      <div className="flex items-center justify-between">
                        <div>
                          <p className="font-medium text-gray-900">{backlink.source}</p>
                          <p className="text-sm text-gray-600">{backlink.url}</p>
                        </div>
                        <div className="text-right">
                          <Badge className={backlink.status === 'active' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}>
                            {backlink.status}
                          </Badge>
                          <p className="text-sm text-gray-600 mt-1">DA: {backlink.authority}</p>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>

      {/* Content Optimizer Modal */}
      <Dialog open={contentOptimizerModalOpen} onOpenChange={setContentOptimizerModalOpen}>
        <DialogContent className="sm:max-w-3xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center">
              <FileText className="mr-2 h-5 w-5 text-blue-600" />
              Content Optimizer
            </DialogTitle>
            <DialogDescription>
              Optimize your content for better SEO performance
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 mt-4">
            <div>
              <Label htmlFor="content-url" className="text-sm font-medium text-gray-700 mb-2 block">
                Page URL (Optional)
              </Label>
              <Input
                id="content-url"
                placeholder="https://example.com/page"
                value={contentOptimizerUrl}
                onChange={(e) => setContentOptimizerUrl(e.target.value)}
              />
            </div>

            <div>
              <Label htmlFor="content-text" className="text-sm font-medium text-gray-700 mb-2 block">
                Content Text
              </Label>
              <Textarea
                id="content-text"
                placeholder="Paste your content here to analyze..."
                value={contentOptimizerText}
                onChange={(e) => setContentOptimizerText(e.target.value)}
                rows={6}
              />
            </div>

            <Button 
              onClick={optimizeContent}
              disabled={isContentOptimizing || (!contentOptimizerText.trim() && !contentOptimizerUrl.trim())}
              className="w-full"
            >
              {isContentOptimizing ? (
                <>
                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                  Optimizing...
                </>
              ) : (
                <>
                  <Brain className="mr-2 h-4 w-4" />
                  Optimize Content
                </>
              )}
            </Button>

            {contentOptimizationResults && (
              <div className="space-y-4">
                <div className="p-4 bg-blue-50 rounded-lg">
                  <h4 className="font-medium text-blue-900 mb-2">SEO Score: {contentOptimizationResults.score}/100</h4>
                  <div className="w-full bg-blue-200 rounded-full h-2">
                    <div 
                      className="bg-blue-600 h-2 rounded-full" 
                      style={{ width: `${contentOptimizationResults.score}%` }}
                    ></div>
                  </div>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Suggestions</h4>
                  <ul className="space-y-1">
                    {contentOptimizationResults.suggestions.map((suggestion: string, index: number) => (
                      <li key={index} className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-0.5 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{suggestion}</span>
                      </li>
                    ))}
                  </ul>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Keyword Density</h4>
                  <div className="space-y-1">
                    {Object.entries(contentOptimizationResults.keywordDensity).map(([keyword, density]: [string, any]) => (
                      <div key={keyword} className="flex justify-between text-sm">
                        <span className="text-gray-700">{keyword}:</span>
                        <span className="font-medium">{density}%</span>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>

      {/* Site Auditor Modal */}
      <Dialog open={siteAuditModalOpen} onOpenChange={setSiteAuditModalOpen}>
        <DialogContent className="sm:max-w-3xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center">
              <Globe className="mr-2 h-5 w-5 text-blue-600" />
              Site Auditor
            </DialogTitle>
            <DialogDescription>
              Comprehensive SEO audit of your website
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 mt-4">
            <div>
              <Label htmlFor="site-audit-url" className="text-sm font-medium text-gray-700 mb-2 block">
                Website URL *
              </Label>
              <Input
                id="site-audit-url"
                placeholder="https://example.com"
                value={siteAuditUrl}
                onChange={(e) => setSiteAuditUrl(e.target.value)}
              />
            </div>

            <Button 
              onClick={auditSite}
              disabled={isSiteAuditing || !siteAuditUrl.trim()}
              className="w-full"
            >
              {isSiteAuditing ? (
                <>
                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                  Auditing Site...
                </>
              ) : (
                <>
                  <Search className="mr-2 h-4 w-4" />
                  Start Site Audit
                </>
              )}
            </Button>

            {siteAuditResults && (
              <div className="space-y-4">
                <div className="p-4 bg-blue-50 rounded-lg">
                  <h4 className="font-medium text-blue-900 mb-2">Overall SEO Score: {siteAuditResults.overallScore}/100</h4>
                  <div className="w-full bg-blue-200 rounded-full h-2">
                    <div 
                      className="bg-blue-600 h-2 rounded-full" 
                      style={{ width: `${siteAuditResults.overallScore}%` }}
                    ></div>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  {Object.entries(siteAuditResults.categories).map(([category, data]: [string, any]) => (
                    <Card key={category}>
                      <CardHeader className="pb-3">
                        <CardTitle className="text-lg capitalize">{category}</CardTitle>
                        <div className="flex items-center">
                          <span className="text-2xl font-bold mr-2">{data.score}</span>
                          <span className="text-sm text-gray-600">/100</span>
                        </div>
                      </CardHeader>
                      <CardContent>
                        <div className="space-y-2">
                          <div>
                            <p className="text-sm font-medium text-red-600">Issues:</p>
                            <ul className="text-xs text-gray-600">
                              {data.issues.map((issue: string, index: number) => (
                                <li key={index}>• {issue}</li>
                              ))}
                            </ul>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-green-600">Fixes:</p>
                            <ul className="text-xs text-gray-600">
                              {data.fixes.map((fix: string, index: number) => (
                                <li key={index}>• {fix}</li>
                              ))}
                            </ul>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Recommendations</h4>
                  <ul className="space-y-1">
                    {siteAuditResults.recommendations.map((rec: string, index: number) => (
                      <li key={index} className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-0.5 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{rec}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>

      {/* Competitor Analysis Modal */}
      <Dialog open={competitorModalOpen} onOpenChange={setCompetitorModalOpen}>
        <DialogContent className="sm:max-w-3xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center">
              <Search className="mr-2 h-5 w-5 text-blue-600" />
              Competitor Analysis
            </DialogTitle>
            <DialogDescription>
              Analyze your competitor's SEO strategy
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 mt-4">
            <div>
              <Label htmlFor="competitor-url" className="text-sm font-medium text-gray-700 mb-2 block">
                Competitor URL *
              </Label>
              <Input
                id="competitor-url"
                placeholder="https://competitor.com"
                value={competitorUrl}
                onChange={(e) => setCompetitorUrl(e.target.value)}
              />
            </div>

            <Button 
              onClick={analyzeCompetitor}
              disabled={isCompetitorAnalyzing || !competitorUrl.trim()}
              className="w-full"
            >
              {isCompetitorAnalyzing ? (
                <>
                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                  Analyzing Competitor...
                </>
              ) : (
                <>
                  <Search className="mr-2 h-4 w-4" />
                  Analyze Competitor
                </>
              )}
            </Button>

            {competitorResults && (
              <div className="space-y-4">
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                  <Card>
                    <CardContent className="p-4 text-center">
                      <p className="text-sm text-gray-600">Domain Authority</p>
                      <p className="text-2xl font-bold text-blue-600">{competitorResults.overview.domainAuthority}</p>
                    </CardContent>
                  </Card>
                  <Card>
                    <CardContent className="p-4 text-center">
                      <p className="text-sm text-gray-600">Organic Keywords</p>
                      <p className="text-2xl font-bold text-green-600">{competitorResults.overview.organicKeywords}</p>
                    </CardContent>
                  </Card>
                  <Card>
                    <CardContent className="p-4 text-center">
                      <p className="text-sm text-gray-600">Monthly Traffic</p>
                      <p className="text-2xl font-bold text-purple-600">{competitorResults.overview.monthlyTraffic}</p>
                    </CardContent>
                  </Card>
                  <Card>
                    <CardContent className="p-4 text-center">
                      <p className="text-sm text-gray-600">Backlinks</p>
                      <p className="text-2xl font-bold text-orange-600">{competitorResults.overview.backlinks}</p>
                    </CardContent>
                  </Card>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Top Keywords</h4>
                  <div className="space-y-2">
                    {competitorResults.topKeywords.map((keyword: any, index: number) => (
                      <div key={index} className="flex items-center justify-between p-2 border rounded">
                        <span className="font-medium">{keyword.keyword}</span>
                        <div className="flex items-center space-x-4">
                          <span className="text-sm text-gray-600">Pos: #{keyword.position}</span>
                          <span className="text-sm text-gray-600">Vol: {keyword.volume.toLocaleString()}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <Card>
                    <CardHeader>
                      <CardTitle className="text-green-600">Strengths</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <ul className="space-y-1">
                        {competitorResults.strengths.map((strength: string, index: number) => (
                          <li key={index} className="flex items-start">
                            <Check className="h-4 w-4 text-green-600 mr-2 mt-0.5 flex-shrink-0" />
                            <span className="text-sm text-gray-700">{strength}</span>
                          </li>
                        ))}
                      </ul>
                    </CardContent>
                  </Card>

                  <Card>
                    <CardHeader>
                      <CardTitle className="text-red-600">Weaknesses</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <ul className="space-y-1">
                        {competitorResults.weaknesses.map((weakness: string, index: number) => (
                          <li key={index} className="flex items-start">
                            <X className="h-4 w-4 text-red-600 mr-2 mt-0.5 flex-shrink-0" />
                            <span className="text-sm text-gray-700">{weakness}</span>
                          </li>
                        ))}
                      </ul>
                    </CardContent>
                  </Card>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Opportunities</h4>
                  <ul className="space-y-1">
                    {competitorResults.opportunities.map((opportunity: string, index: number) => (
                      <li key={index} className="flex items-start">
                        <Target className="h-4 w-4 text-blue-600 mr-2 mt-0.5 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{opportunity}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>
    </div>
  )
}