'use client'

import { useState } from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { 
  Users, 
  TrendingUp, 
  Eye, 
  Share2, 
  Search,
  Brain,
  Plus,
  ArrowUpRight,
  ArrowDownRight
} from 'lucide-react'

interface OverviewTabProps {
  onTabChange?: (tab: string) => void
}

export function OverviewTab({ onTabChange }: OverviewTabProps) {
  const [activeTab, setActiveTab] = useState('activity')

  const handleNewCampaign = () => {
    console.log('Creating new campaign...')
    // Navigate to AI tools tab for campaign creation
    if (onTabChange) {
      onTabChange('ai-tools')
    }
  }

  const handleAIAssistant = () => {
    console.log('Opening AI Assistant...')
    // Navigate to AI tools tab
    if (onTabChange) {
      onTabChange('ai-tools')
    }
  }

  const handleStatClick = (statTitle: string) => {
    console.log(`Clicked on ${statTitle}`)
    // Here you could navigate to detailed analytics for that stat
    alert(`Detailed ${statTitle} analytics coming soon!`)
  }

  const handleActivityClick = (activity: any) => {
    console.log('Clicked on activity:', activity)
    alert(`Activity details for ${activity.platform} post coming soon!`)
  }

  const handleRankingClick = (ranking: any) => {
    console.log('Clicked on ranking:', ranking)
    alert(`Detailed ranking analysis for "${ranking.keyword}" coming soon!`)
  }

  const handleCreateCampaign = () => {
    console.log('Creating first campaign...')
    // Navigate to AI tools tab for campaign creation
    if (onTabChange) {
      onTabChange('ai-tools')
    }
  }
  const stats = [
    {
      title: "Total Followers",
      value: "124.5K",
      change: "+12.5%",
      trend: "up",
      icon: Users,
      color: "text-blue-600"
    },
    {
      title: "Engagement Rate",
      value: "8.2%",
      change: "+2.1%",
      trend: "up", 
      icon: TrendingUp,
      color: "text-green-600"
    },
    {
      title: "Website Traffic",
      value: "45.2K",
      change: "-5.3%",
      trend: "down",
      icon: Eye,
      color: "text-purple-600"
    },
    {
      title: "Social Shares",
      value: "8,942",
      change: "+18.2%",
      trend: "up",
      icon: Share2,
      color: "text-orange-600"
    }
  ]

  const recentActivity = [
    {
      platform: "Instagram",
      action: "Post published",
      content: "New product launch 🚀",
      time: "2 hours ago",
      performance: "High"
    },
    {
      platform: "Twitter",
      action: "Campaign started",
      content: "Summer sale promotion",
      time: "4 hours ago", 
      performance: "Medium"
    },
    {
      platform: "LinkedIn",
      action: "Article published",
      content: "Industry insights 2024",
      time: "6 hours ago",
      performance: "High"
    }
  ]

  const seoRankings = [
    {
      keyword: "social media marketing",
      position: 12,
      change: "+3",
      url: "/blog/social-media-marketing"
    },
    {
      keyword: "ai content generator",
      position: 8,
      change: "+5",
      url: "/tools/ai-content-generator"
    },
    {
      keyword: "instagram growth",
      position: 15,
      change: "-2",
      url: "/services/instagram-growth"
    }
  ]

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Dashboard Overview</h1>
          <p className="text-gray-600 mt-2">Monitor your social media and SEO performance</p>
        </div>
        <div className="flex space-x-3">
          <Button 
            onClick={handleNewCampaign}
            className="hover:scale-105 transition-transform duration-200"
          >
            <Plus className="mr-2 h-4 w-4" />
            New Campaign
          </Button>
          <Button 
            variant="outline" 
            onClick={handleAIAssistant}
            className="hover:scale-105 transition-transform duration-200"
          >
            <Brain className="mr-2 h-4 w-4" />
            AI Assistant
          </Button>
        </div>
      </div>

      {/* Stats Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {stats.map((stat, index) => {
          const Icon = stat.icon
          return (
            <Card 
              key={index} 
              className="cursor-pointer hover:shadow-lg transition-shadow duration-200"
              onClick={() => handleStatClick(stat.title)}
            >
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-gray-600">{stat.title}</p>
                    <p className="text-2xl font-bold text-gray-900 mt-1">{stat.value}</p>
                  </div>
                  <div className={`p-3 bg-gray-50 rounded-lg ${stat.color}`}>
                    <Icon className="h-6 w-6" />
                  </div>
                </div>
                <div className="flex items-center mt-4">
                  {stat.trend === 'up' ? (
                    <ArrowUpRight className="h-4 w-4 text-green-600 mr-1" />
                  ) : (
                    <ArrowDownRight className="h-4 w-4 text-red-600 mr-1" />
                  )}
                  <span className={`text-sm font-medium ${
                    stat.trend === 'up' ? 'text-green-600' : 'text-red-600'
                  }`}>
                    {stat.change}
                  </span>
                  <span className="text-sm text-gray-500 ml-2">vs last month</span>
                </div>
              </CardContent>
            </Card>
          )
        })}
      </div>

      <Tabs defaultValue="activity" value={activeTab} onValueChange={setActiveTab} className="space-y-6">
        <TabsList>
          <TabsTrigger value="activity">Recent Activity</TabsTrigger>
          <TabsTrigger value="seo">SEO Rankings</TabsTrigger>
          <TabsTrigger value="campaigns">Active Campaigns</TabsTrigger>
        </TabsList>

        <TabsContent value="activity">
          <Card>
            <CardHeader>
              <CardTitle>Recent Activity</CardTitle>
              <CardDescription>Latest social media and content activities</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {recentActivity.map((activity, index) => (
                  <div 
                    key={index} 
                    className="flex items-center justify-between p-4 bg-gray-50 rounded-lg cursor-pointer hover:bg-gray-100 transition-colors duration-200"
                    onClick={() => handleActivityClick(activity)}
                  >
                    <div className="flex items-center space-x-4">
                      <div className="w-10 h-10 bg-gradient-to-r from-purple-600 to-blue-600 rounded-lg flex items-center justify-center">
                        <span className="text-white font-bold text-sm">
                          {activity.platform.charAt(0)}
                        </span>
                      </div>
                      <div>
                        <p className="font-medium text-gray-900">{activity.action}</p>
                        <p className="text-sm text-gray-600">{activity.content}</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <Badge variant={activity.performance === 'High' ? 'default' : 'secondary'}>
                        {activity.performance}
                      </Badge>
                      <p className="text-xs text-gray-500 mt-1">{activity.time}</p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="seo">
          <Card>
            <CardHeader>
              <CardTitle>Keyword Rankings</CardTitle>
              <CardDescription>Track your SEO performance</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {seoRankings.map((ranking, index) => (
                  <div 
                    key={index} 
                    className="flex items-center justify-between p-4 bg-gray-50 rounded-lg cursor-pointer hover:bg-gray-100 transition-colors duration-200"
                    onClick={() => handleRankingClick(ranking)}
                  >
                    <div className="flex items-center space-x-4">
                      <Search className="h-5 w-5 text-gray-400" />
                      <div>
                        <p className="font-medium text-gray-900">{ranking.keyword}</p>
                        <p className="text-sm text-gray-600">{ranking.url}</p>
                      </div>
                    </div>
                    <div className="flex items-center space-x-4">
                      <div className="text-right">
                        <p className="text-lg font-bold text-gray-900">#{ranking.position}</p>
                        <div className="flex items-center">
                          {ranking.change.startsWith('+') ? (
                            <ArrowUpRight className="h-3 w-3 text-green-600 mr-1" />
                          ) : (
                            <ArrowDownRight className="h-3 w-3 text-red-600 mr-1" />
                          )}
                          <span className={`text-sm ${
                            ranking.change.startsWith('+') ? 'text-green-600' : 'text-red-600'
                          }`}>
                            {ranking.change}
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="campaigns">
          <Card>
            <CardHeader>
              <CardTitle>Active Campaigns</CardTitle>
              <CardDescription>Monitor your running campaigns</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8">
                <Brain className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No active campaigns</h3>
                <p className="text-gray-600 mb-4">Create your first campaign to get started</p>
                <Button 
                onClick={handleCreateCampaign}
                className="hover:scale-105 transition-transform duration-200"
              >
                  <Plus className="mr-2 h-4 w-4" />
                  Create Campaign
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
}