'use client'

import { useState, useEffect } from 'react'
import dynamic from 'next/dynamic'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Switch } from '@/components/ui/switch'
import { Label } from '@/components/ui/label'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Activity, Eye, TrendingUp, MapPin, Clock, Users, Wifi, WifiOff } from 'lucide-react'
import { useWebSocket } from '@/hooks/useWebSocket'

// Dynamically import map components to avoid SSR issues
const MapContainer = dynamic(
  () => import('react-leaflet').then((mod) => mod.MapContainer),
  { ssr: false }
)

const TileLayer = dynamic(
  () => import('react-leaflet').then((mod) => mod.TileLayer),
  { ssr: false }
)

const Marker = dynamic(
  () => import('react-leaflet').then((mod) => mod.Marker),
  { ssr: false }
)

const Popup = dynamic(
  () => import('react-leaflet').then((mod) => mod.Popup),
  { ssr: false }
)

interface LocationData {
  id: string
  lat: number
  lng: number
  city: string
  country: string
  views: number
  ranking: number
  platform: string
  trend: 'up' | 'down' | 'stable'
  growth: number
  timestamp: string
}

interface LiveFeedItem {
  id: string
  type: 'new_view' | 'ranking_change' | 'trend_spike'
  location: string
  message: string
  timestamp: string
  importance: 'high' | 'medium' | 'low'
}

export function MapTab() {
  const [locationData, setLocationData] = useState<LocationData[]>([])
  const [liveFeed, setLiveFeed] = useState<LiveFeedItem[]>([])
  const [isLive, setIsLive] = useState(true)
  const [selectedMetric, setSelectedMetric] = useState<'views' | 'ranking' | 'growth'>('views')
  const [isLoading, setIsLoading] = useState(true)
  
  // WebSocket hook for real-time updates
  const { isConnected, locationUpdate, liveFeedEvent } = useWebSocket('map-updates')

  // Handle WebSocket updates
  useEffect(() => {
    if (!isLive) return

    // Handle location updates from WebSocket
    if (locationUpdate) {
      setLocationData(prev => {
        const existingIndex = prev.findIndex(loc => loc.city === locationUpdate.city)
        if (existingIndex >= 0) {
          // Update existing location
          const updated = [...prev]
          updated[existingIndex] = locationUpdate
          return updated
        } else {
          // Add new location
          return [...prev, locationUpdate].slice(0, 20) // Keep max 20 locations
        }
      })
    }

    // Handle live feed events from WebSocket
    if (liveFeedEvent) {
      setLiveFeed(prev => [liveFeedEvent, ...prev].slice(0, 15))
    }
  }, [locationUpdate, liveFeedEvent, isLive])

  // Fetch location data from API
  useEffect(() => {
    const fetchLocationData = async () => {
      try {
        const response = await fetch(`/api/map-data?metric=${selectedMetric}&limit=20`)
        const result = await response.json()
        
        if (result.success) {
          setLocationData(result.data)
        }
      } catch (error) {
        console.error('Error fetching location data:', error)
      }
    }

    const fetchLiveFeed = async () => {
      try {
        const response = await fetch('/api/live-feed?limit=10')
        const result = await response.json()
        
        if (result.success) {
          setLiveFeed(result.data)
        }
      } catch (error) {
        console.error('Error fetching live feed:', error)
      }
    }

    const loadData = async () => {
      setIsLoading(true)
      await Promise.all([fetchLocationData(), fetchLiveFeed()])
      setIsLoading(false)
    }

    loadData()
  }, [selectedMetric])

  // WebSocket updates are handled in the effect above

  const getMarkerColor = (trend: 'up' | 'down' | 'stable') => {
    switch (trend) {
      case 'up': return '#22c55e'
      case 'down': return '#ef4444'
      default: return '#3b82f6'
    }
  }

  const formatTimeAgo = (timestamp: string) => {
    const now = new Date()
    const time = new Date(timestamp)
    const diff = Math.floor((now.getTime() - time.getTime()) / 1000)
    
    if (diff < 60) return 'just now'
    if (diff < 3600) return `${Math.floor(diff / 60)}m ago`
    return `${Math.floor(diff / 3600)}h ago`
  }

  const getImportanceColor = (importance: 'high' | 'medium' | 'low') => {
    switch (importance) {
      case 'high': return 'bg-red-500'
      case 'medium': return 'bg-yellow-500'
      default: return 'bg-green-500'
    }
  }

  if (isLoading) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold text-gray-900">Live Rankings Map</h1>
            <p className="text-gray-600 mt-1">Loading map data...</p>
          </div>
        </div>
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          <div className="lg:col-span-2 h-96 bg-gray-100 rounded-lg animate-pulse"></div>
          <div className="space-y-4">
            <div className="h-32 bg-gray-100 rounded-lg animate-pulse"></div>
            <div className="h-64 bg-gray-100 rounded-lg animate-pulse"></div>
          </div>
        </div>
      </div>
    )
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h1 className="text-2xl sm:text-3xl font-bold text-gray-900">Live Rankings Map</h1>
          <p className="text-gray-600 mt-1">Real-time geographic insights and rankings</p>
        </div>
        <div className="flex items-center space-x-4">
          {/* Connection Status */}
          <div className="flex items-center space-x-2">
            {isConnected ? (
              <>
                <Wifi className="h-4 w-4 text-green-500" />
                <span className="text-xs text-green-600 font-medium">Connected</span>
              </>
            ) : (
              <>
                <WifiOff className="h-4 w-4 text-red-500" />
                <span className="text-xs text-red-600 font-medium">Disconnected</span>
              </>
            )}
          </div>
          
          <div className="flex items-center space-x-2">
            <Switch
              id="live-mode"
              checked={isLive}
              onCheckedChange={setIsLive}
            />
            <Label htmlFor="live-mode" className="text-sm font-medium">
              Live Updates
            </Label>
            {isLive && (
              <div className="flex items-center space-x-1">
                <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                <span className="text-xs text-gray-500">LIVE</span>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <Eye className="h-4 w-4 text-blue-500" />
              <div>
                <p className="text-xs text-gray-500">Total Views</p>
                <p className="text-lg font-bold">
                  {locationData.reduce((sum, loc) => sum + loc.views, 0).toLocaleString()}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <MapPin className="h-4 w-4 text-green-500" />
              <div>
                <p className="text-xs text-gray-500">Active Locations</p>
                <p className="text-lg font-bold">{locationData.length}</p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <TrendingUp className="h-4 w-4 text-purple-500" />
              <div>
                <p className="text-xs text-gray-500">Avg Growth</p>
                <p className="text-lg font-bold">
                  {(locationData.reduce((sum, loc) => sum + loc.growth, 0) / locationData.length).toFixed(1)}%
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <Activity className="h-4 w-4 text-orange-500" />
              <div>
                <p className="text-xs text-gray-500">Live Events</p>
                <p className="text-lg font-bold">{liveFeed.length}</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Main Content */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Map Section */}
        <div className="lg:col-span-2">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle className="text-lg">Geographic Distribution</CardTitle>
                  <CardDescription>Click markers for detailed insights</CardDescription>
                </div>
                <Tabs value={selectedMetric} onValueChange={(value) => setSelectedMetric(value as any)}>
                  <TabsList className="grid w-full grid-cols-3">
                    <TabsTrigger value="views" className="text-xs">Views</TabsTrigger>
                    <TabsTrigger value="ranking" className="text-xs">Ranking</TabsTrigger>
                    <TabsTrigger value="growth" className="text-xs">Growth</TabsTrigger>
                  </TabsList>
                </Tabs>
              </div>
            </CardHeader>
            <CardContent>
              <div className="h-96 bg-gray-50 rounded-lg flex items-center justify-center">
                <div className="text-center">
                  <MapPin className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">Interactive Map</h3>
                  <p className="text-gray-500 mb-4">Map visualization would be displayed here</p>
                  <div className="grid grid-cols-2 gap-4 max-w-xs mx-auto">
                    {locationData.map((location) => (
                      <div key={location.id} className="bg-white p-3 rounded-lg border">
                        <div className="flex items-center justify-between mb-2">
                          <span className="font-medium text-sm">{location.city}</span>
                          <div 
                            className="w-3 h-3 rounded-full"
                            style={{ backgroundColor: getMarkerColor(location.trend) }}
                          ></div>
                        </div>
                        <div className="text-xs text-gray-500">
                          {selectedMetric === 'views' && `${location.views.toLocaleString()} views`}
                          {selectedMetric === 'ranking' && `Rank #${location.ranking}`}
                          {selectedMetric === 'growth' && `${location.growth > 0 ? '+' : ''}${location.growth.toFixed(1)}%`}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Live Feed Section */}
        <div className="space-y-6">
          {/* Live Feed */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg flex items-center">
                <Activity className="h-5 w-5 mr-2 text-orange-500" />
                Live Feed
              </CardTitle>
              <CardDescription>Real-time ranking updates</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3 max-h-80 overflow-y-auto">
                {liveFeed.map((item) => (
                  <div key={item.id} className="flex items-start space-x-3 p-3 bg-gray-50 rounded-lg">
                    <div className={`w-2 h-2 rounded-full mt-2 ${getImportanceColor(item.importance)}`}></div>
                    <div className="flex-1 min-w-0">
                      <p className="text-sm font-medium text-gray-900 truncate">{item.message}</p>
                      <p className="text-xs text-gray-500 flex items-center mt-1">
                        <MapPin className="h-3 w-3 mr-1" />
                        {item.location}
                        <span className="mx-1">•</span>
                        <Clock className="h-3 w-3 mr-1" />
                        {formatTimeAgo(item.timestamp)}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Top Locations */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg flex items-center">
                <TrendingUp className="h-5 w-5 mr-2 text-green-500" />
                Top Locations
              </CardTitle>
              <CardDescription>Best performing regions</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {locationData
                  .sort((a, b) => b.views - a.views)
                  .slice(0, 5)
                  .map((location, index) => (
                    <div key={location.id} className="flex items-center justify-between">
                      <div className="flex items-center space-x-3">
                        <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                          <span className="text-xs font-bold text-blue-600">{index + 1}</span>
                        </div>
                        <div>
                          <p className="text-sm font-medium">{location.city}</p>
                          <p className="text-xs text-gray-500">{location.platform}</p>
                        </div>
                      </div>
                      <div className="text-right">
                        <p className="text-sm font-bold">{location.views.toLocaleString()}</p>
                        <div className="flex items-center space-x-1">
                          {location.trend === 'up' && <TrendingUp className="h-3 w-3 text-green-500" />}
                          {location.trend === 'down' && <TrendingUp className="h-3 w-3 text-red-500 rotate-180" />}
                          <span className={`text-xs ${location.trend === 'up' ? 'text-green-500' : location.trend === 'down' ? 'text-red-500' : 'text-gray-500'}`}>
                            {location.growth > 0 ? '+' : ''}{location.growth.toFixed(1)}%
                          </span>
                        </div>
                      </div>
                    </div>
                  ))}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  )
}