'use client'

import { useState } from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { 
  BarChart3, 
  TrendingUp, 
  Users, 
  Eye, 
  Heart,
  MessageCircle,
  Share,
  Download,
  Calendar,
  ArrowUpRight,
  ArrowDownRight
} from 'lucide-react'

export function AnalyticsTab() {
  const [selectedDateRange, setSelectedDateRange] = useState('7d')

  const handleDateRange = () => {
    console.log('Date range selector clicked')
    alert('Date range selector would open a calendar modal here')
  }

  const handleExportReport = () => {
    console.log('Export report clicked')
    alert('Report export functionality would download a PDF/Excel report here')
  }

  const handleStatClick = (statTitle: string) => {
    console.log(`Clicked on ${statTitle} stat`)
    alert(`Detailed ${statTitle} analytics would open here`)
  }

  const handlePlatformClick = (platform: string) => {
    console.log(`Clicked on ${platform} performance`)
    alert(`Detailed ${platform} performance analytics would open here`)
  }

  const handlePostClick = (post: any) => {
    console.log('Clicked on post:', post)
    alert(`Detailed post analytics for "${post.content}" would open here`)
  }

  const overviewStats = [
    {
      title: "Total Reach",
      value: "2.4M",
      change: "+18.2%",
      trend: "up",
      icon: Eye,
      color: "text-blue-600"
    },
    {
      title: "Engagement Rate",
      value: "12.5%",
      change: "+3.1%",
      trend: "up",
      icon: Heart,
      color: "text-red-600"
    },
    {
      title: "New Followers",
      value: "45.2K",
      change: "+22.5%",
      trend: "up",
      icon: Users,
      color: "text-green-600"
    },
    {
      title: "Total Shares",
      value: "128K",
      change: "+8.7%",
      trend: "up",
      icon: Share,
      color: "text-purple-600"
    }
  ]

  const platformPerformance = [
    {
      platform: "Instagram",
      followers: "45.2K",
      engagement: "8.2%",
      reach: "1.2M",
      growth: "+15.3%"
    },
    {
      platform: "Twitter",
      followers: "22.8K",
      engagement: "6.5%",
      reach: "680K",
      growth: "+8.7%"
    },
    {
      platform: "Facebook",
      followers: "38.5K",
      engagement: "5.8%",
      reach: "920K",
      growth: "+12.1%"
    },
    {
      platform: "LinkedIn",
      followers: "18.0K",
      engagement: "9.2%",
      reach: "420K",
      growth: "+18.9%"
    }
  ]

  const topPosts = [
    {
      id: 1,
      platform: "Instagram",
      content: "Product launch announcement 🚀",
      engagement: "15.2K",
      reach: "125K",
      performance: "Excellent"
    },
    {
      id: 2,
      platform: "Twitter",
      content: "Industry insights thread",
      engagement: "8.7K",
      reach: "89K",
      performance: "Good"
    },
    {
      id: 3,
      platform: "LinkedIn",
      content: "Company milestone celebration",
      engagement: "6.3K",
      reach: "67K",
      performance: "Good"
    }
  ]

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Analytics Dashboard</h1>
          <p className="text-gray-600 mt-2">Comprehensive insights into your social media and SEO performance</p>
        </div>
        <div className="flex space-x-3">
          <Button 
            variant="outline"
            onClick={handleDateRange}
            className="hover:scale-105 transition-transform"
          >
            <Calendar className="mr-2 h-4 w-4" />
            Date Range
          </Button>
          <Button 
            onClick={handleExportReport}
            className="hover:scale-105 transition-transform"
          >
            <Download className="mr-2 h-4 w-4" />
            Export Report
          </Button>
        </div>
      </div>

      {/* Overview Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {overviewStats.map((stat, index) => {
          const Icon = stat.icon
          return (
            <Card 
              key={index}
              className="cursor-pointer hover:shadow-lg transition-shadow duration-200"
              onClick={() => handleStatClick(stat.title)}
            >
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-gray-600">{stat.title}</p>
                    <p className="text-2xl font-bold text-gray-900 mt-1">{stat.value}</p>
                  </div>
                  <div className={`p-3 bg-gray-50 rounded-lg ${stat.color}`}>
                    <Icon className="h-6 w-6" />
                  </div>
                </div>
                <div className="flex items-center mt-4">
                  {stat.trend === 'up' ? (
                    <ArrowUpRight className="h-4 w-4 text-green-600 mr-1" />
                  ) : (
                    <ArrowDownRight className="h-4 w-4 text-red-600 mr-1" />
                  )}
                  <span className={`text-sm font-medium ${
                    stat.trend === 'up' ? 'text-green-600' : 'text-red-600'
                  }`}>
                    {stat.change}
                  </span>
                  <span className="text-sm text-gray-500 ml-2">vs last month</span>
                </div>
              </CardContent>
            </Card>
          )
        })}
      </div>

      <Tabs defaultValue="overview" className="space-y-6">
        <TabsList>
          <TabsTrigger value="overview">Overview</TabsTrigger>
          <TabsTrigger value="social">Social Media</TabsTrigger>
          <TabsTrigger value="seo">SEO Analytics</TabsTrigger>
          <TabsTrigger value="audience">Audience Insights</TabsTrigger>
        </TabsList>

        <TabsContent value="overview">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Platform Performance</CardTitle>
                <CardDescription>How each platform is performing</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {platformPerformance.map((platform, index) => (
                    <div 
                      key={index} 
                      className="flex items-center justify-between p-4 bg-gray-50 rounded-lg cursor-pointer hover:bg-gray-100 transition-colors"
                      onClick={() => handlePlatformClick(platform.platform)}
                    >
                      <div>
                        <h4 className="font-medium text-gray-900">{platform.platform}</h4>
                        <p className="text-sm text-gray-600">{platform.followers} followers</p>
                      </div>
                      <div className="text-right">
                        <p className="text-sm font-medium text-gray-900">{platform.engagement} engagement</p>
                        <p className="text-xs text-green-600">{platform.growth} growth</p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Top Performing Posts</CardTitle>
                <CardDescription>Your most engaging content</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {topPosts.map((post) => (
                    <div 
                      key={post.id} 
                      className="border rounded-lg p-4 cursor-pointer hover:bg-gray-50 transition-colors"
                      onClick={() => handlePostClick(post)}
                    >
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <p className="font-medium text-gray-900 mb-1">{post.content}</p>
                          <p className="text-sm text-gray-600">{post.platform}</p>
                          <div className="flex items-center space-x-4 mt-2 text-sm text-gray-500">
                            <div className="flex items-center">
                              <Heart className="h-3 w-3 mr-1" />
                              {post.engagement}
                            </div>
                            <div className="flex items-center">
                              <Eye className="h-3 w-3 mr-1" />
                              {post.reach}
                            </div>
                          </div>
                        </div>
                        <Badge variant={post.performance === 'Excellent' ? 'default' : 'secondary'}>
                          {post.performance}
                        </Badge>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="social">
          <Card>
            <CardHeader>
              <CardTitle>Social Media Analytics</CardTitle>
              <CardDescription>Detailed social media performance metrics</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-12">
                <BarChart3 className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">Advanced Analytics</h3>
                <p className="text-gray-600 mb-4">Detailed social media analytics and insights</p>
                <Button 
                  onClick={() => alert('Advanced social media analytics would open here')}
                  className="hover:scale-105 transition-transform"
                >
                  View Detailed Analytics
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="seo">
          <Card>
            <CardHeader>
              <CardTitle>SEO Performance</CardTitle>
              <CardDescription>Search engine optimization metrics</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-12">
                <TrendingUp className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">SEO Analytics</h3>
                <p className="text-gray-600 mb-4">Keyword rankings, organic traffic, and SEO insights</p>
                <Button 
                  onClick={() => alert('Detailed SEO analytics would open here')}
                  className="hover:scale-105 transition-transform"
                >
                  View SEO Details
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="audience">
          <Card>
            <CardHeader>
              <CardTitle>Audience Insights</CardTitle>
              <CardDescription>Understand your audience demographics and behavior</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-12">
                <Users className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">Audience Analytics</h3>
                <p className="text-gray-600 mb-4">Demographics, interests, and engagement patterns</p>
                <Button 
                  onClick={() => alert('Detailed audience insights would open here')}
                  className="hover:scale-105 transition-transform"
                >
                  View Audience Details
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
}