'use client'

import { useState } from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Textarea } from '@/components/ui/textarea'
import { Badge } from '@/components/ui/badge'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Label } from '@/components/ui/label'
import { 
  Brain, 
  FileText, 
  Image as ImageIcon, 
  Video, 
  MessageSquare,
  TrendingUp,
  Zap,
  Target,
  Users,
  Search,
  Globe,
  Sparkles,
  Download,
  Share,
  Send
} from 'lucide-react'
import { ModernChatbot } from '@/components/ui/modern-chatbot'
import { toast } from 'sonner'

interface AiToolsTabProps {
  onTabChange?: (tab: string) => void
}

export function AiToolsTab({ onTabChange }: AiToolsTabProps) {
  const [selectedTool, setSelectedTool] = useState('content')
  const [isGenerating, setIsGenerating] = useState(false)
  const [generatedContent, setGeneratedContent] = useState('')
  const [contentType, setContentType] = useState('Blog Post')
  const [topic, setTopic] = useState('')
  const [tone, setTone] = useState('')
  const [length, setLength] = useState('')
  const [additionalInstructions, setAdditionalInstructions] = useState('')
  const [error, setError] = useState('')
  const [generatedImage, setGeneratedImage] = useState('')
  const [imagePrompt, setImagePrompt] = useState('')
  const [imageStyle, setImageStyle] = useState('realistic')
  const [generatedVideo, setGeneratedVideo] = useState('')
  const [videoPrompt, setVideoPrompt] = useState('')
  const [videoDuration, setVideoDuration] = useState('10')
  const [videoStyle, setVideoStyle] = useState('realistic')
  const [videoFormat, setVideoFormat] = useState('mp4')
  const [chatMessages, setChatMessages] = useState<Array<{role: 'user' | 'assistant', content: string}>>([])
  const [chatInput, setChatInput] = useState('')

  const aiTools = [
    {
      id: 'content',
      name: 'Content Generator',
      description: 'Create SEO-optimized articles and posts',
      icon: FileText,
      color: 'from-blue-600 to-purple-600'
    },
    {
      id: 'image',
      name: 'Image Generator',
      description: 'Generate stunning visuals with AI',
      icon: ImageIcon,
      color: 'from-green-600 to-blue-600'
    },
    {
      id: 'video',
      name: 'AI Video Generator',
      description: 'Generate actual videos from prompts',
      icon: Video,
      color: 'from-purple-600 to-pink-600'
    },
    {
      id: 'chatbot',
      name: 'AI Chatbot',
      description: 'Automated customer engagement',
      icon: MessageSquare,
      color: 'from-orange-600 to-red-600'
    }
  ]

  const contentTemplates = [
    {
      name: 'Blog Post',
      description: 'SEO-optimized blog articles',
      prompt: 'Write a comprehensive blog post about'
    },
    {
      name: 'Social Media Post',
      description: 'Engaging social media content',
      prompt: 'Create an engaging social media post about'
    },
    {
      name: 'Product Description',
      description: 'Compelling product descriptions',
      prompt: 'Write a persuasive product description for'
    },
    {
      name: 'Email Newsletter',
      description: 'Professional email content',
      prompt: 'Create an email newsletter about'
    }
  ]

  const handleRegenerate = () => {
    console.log('Regenerating content...')
    setIsGenerating(true)
    setTimeout(() => {
      setGeneratedContent(`This is REGENERATED AI content that would normally be created using the z-ai-web-dev-sdk. 

The content has been refreshed with new ideas and perspectives while maintaining the same quality standards.

Key improvements in this version:
- Enhanced readability and flow
- Updated insights and perspectives  
- Fresh approach to the topic
- Optimized for current trends
- Improved engagement potential`)
      setIsGenerating(false)
    }, 2000)
  }

  const handleOptimizeSEO = () => {
    console.log('Optimizing for SEO...')
    
    // Show loading toast
    toast.loading('Navigating to SEO optimization...')
    
    // Navigate to SEO tab for detailed optimization
    setTimeout(() => {
      if (onTabChange) {
        onTabChange('seo')
        toast.success('Navigated to SEO optimization tools')
      }
    }, 500)
  }

  const handleExport = () => {
    console.log('Exporting content...')
    alert('Content export functionality would download the content in various formats (PDF, Word, HTML, etc.)')
  }

  const handleShare = () => {
    console.log('Sharing content...')
    alert('Content sharing functionality would allow you to share via email, social media, or generate a shareable link.')
  }

  const handleAnalyzeOptimize = () => {
    console.log('Analyzing and optimizing content...')
    
    // Show loading toast
    toast.loading('Navigating to content analysis...')
    
    // Navigate to SEO tab for optimization
    setTimeout(() => {
      if (onTabChange) {
        onTabChange('seo')
        toast.success('Navigated to content analysis tools')
      }
    }, 500)
  }

  const handleConfigureAutomation = (type: string) => {
    console.log(`Navigating to ${type} automation...`)
    
    // Show loading toast
    toast.loading(`Configuring ${type} automation...`)
    
    // Navigate to the appropriate tab based on the automation type
    setTimeout(() => {
      switch (type) {
        case 'Social Media':
          if (onTabChange) {
            onTabChange('social-media')
            toast.success('Navigated to Social Media automation')
          }
          break
        case 'SEO':
          if (onTabChange) {
            onTabChange('seo')
            toast.success('Navigated to SEO automation')
          }
          break
        case 'Content':
          // Stay on AI tools but switch to content generator
          setSelectedTool('content')
          toast.success('Switched to Content automation tools')
          break
        default:
          console.log(`Unknown automation type: ${type}`)
          toast.error('Unknown automation type')
      }
    }, 500)
  }

  const handleGenerate = async () => {
    if (!topic.trim()) {
      setError('Please enter a topic or keyword')
      return
    }

    setIsGenerating(true)
    setError('')
    
    try {
      console.log('handleGenerate: Starting content generation...')
      console.log('handleGenerate: Topic:', topic)
      console.log('handleGenerate: ContentType:', contentType)
      console.log('handleGenerate: Tone:', tone)
      console.log('handleGenerate: Length:', length)
      
      // Show loading toast
      toast.loading('Generating content with AI...')
      
      const requestData = {
        type: 'content',
        prompt: topic,
        options: {
          contentType: contentType.toLowerCase(),
          tone: tone || 'professional',
          length: length || 'medium',
          additionalInstructions: additionalInstructions || ''
        }
      }
      
      console.log('handleGenerate: Request data:', requestData)
      
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(requestData),
        signal: AbortSignal.timeout(60000) // 60 second timeout
      })

      console.log('handleGenerate: Response status:', response.status)
      console.log('handleGenerate: Response headers:', response.headers)
      
      if (!response.ok) {
        const errorText = await response.text()
        console.error('handleGenerate: Error response text:', errorText)
        const errorData = JSON.parse(errorText)
        console.error('handleGenerate: API error response:', errorData)
        throw new Error(errorData.error || `HTTP ${response.status}: ${response.statusText}`)
      }

      const data = await response.json()
      console.log('handleGenerate: API response data:', data)
      console.log('handleGenerate: Response data type:', typeof data)
      console.log('handleGenerate: Has content property:', 'content' in data)
      console.log('handleGenerate: Content value:', data.content)
      
      if (data.error) {
        console.error('handleGenerate: Error in response data:', data.error)
        throw new Error(data.error)
      }

      if (!data.content) {
        console.error('handleGenerate: No content in response')
        console.error('handleGenerate: Available properties:', Object.keys(data))
        throw new Error('No content generated - API response missing content property')
      }

      console.log('handleGenerate: Content generated successfully, length:', data.content.length)
      setGeneratedContent(data.content)
      
      // Show success toast
      toast.success('Content generated successfully!')
      
    } catch (error) {
      console.error('handleGenerate: Error generating content:', error)
      console.error('handleGenerate: Error type:', typeof error)
      console.error('handleGenerate: Error name:', error instanceof Error ? error.name : 'Unknown')
      console.error('handleGenerate: Error message:', error instanceof Error ? error.message : 'Unknown error')
      
      let errorMessage = 'Failed to generate content'
      
      if (error instanceof Error) {
        if (error.name === 'TimeoutError') {
          errorMessage = 'Request timed out. Please try again.'
        } else if (error.name === 'AbortError') {
          errorMessage = 'Request was cancelled. Please try again.'
        } else {
          errorMessage = error.message
        }
      }
      
      setError(errorMessage)
      
      // Show error toast
      toast.error(errorMessage)
      
      // Fallback to mock content if API fails
      console.log('handleGenerate: Using fallback content generation')
      setGeneratedContent(`# ${contentType.charAt(0).toUpperCase() + contentType.slice(1)}: ${topic}

This is a professionally generated ${contentType} about "${topic}" with a ${tone} tone. The content is optimized for SEO and engagement.

## Introduction

In today's digital landscape, ${topic} has become increasingly important for businesses and individuals alike. Understanding the key aspects and best practices can help you achieve better results and stay ahead of the competition.

## Key Points

• **Strategic Approach**: A well-planned strategy is essential for success with ${topic}
• **Best Practices**: Following industry standards and proven methodologies
• **Implementation**: Practical steps to execute your ${topic} effectively
• **Measurement**: Tracking performance and optimizing for better results

## Detailed Analysis

When it comes to ${topic}, there are several factors to consider. First, you need to understand your target audience and their specific needs. This allows you to tailor your approach and deliver maximum value.

### Key Benefits

1. **Improved Performance**: Better results through optimized strategies
2. **Cost Efficiency**: Maximizing ROI while minimizing expenses
3. **Scalability**: Solutions that grow with your needs
4. **Competitive Advantage**: Standing out in the marketplace

## Implementation Steps

1. **Research and Planning**: Thorough analysis of your requirements
2. **Strategy Development**: Creating a comprehensive plan
3. **Execution**: Implementing your ${topic} strategy
4. **Monitoring**: Tracking progress and making adjustments

## Conclusion

${topic} is a critical component of modern success. By following the strategies and best practices outlined in this ${contentType}, you'll be well-positioned to achieve your goals and drive meaningful results.

---

*This content was generated by AI and optimized for SEO. Contact us for more customized content solutions.*`)
    } finally {
      setIsGenerating(false)
    }
  }

  const handleGenerateImage = async () => {
    if (!imagePrompt.trim()) {
      setError('Please enter an image description')
      return
    }

    setIsGenerating(true)
    setError('')
    
    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'image',
          prompt: imagePrompt,
          options: {
            style: imageStyle,
            size: '1024x1024'
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to generate image')
      }

      const data = await response.json()
      
      if (data.error) {
        throw new Error(data.error)
      }

      setGeneratedImage(data.image || '')
    } catch (error) {
      console.error('Error generating image:', error)
      setError(error instanceof Error ? error.message : 'Failed to generate image')
      // Fallback to placeholder if API fails
      setGeneratedImage('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAwIiBoZWlnaHQ9IjQwMCIgdmlld0JveD0iMCAwIDQwMCA0MDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI0MDAiIGhlaWdodD0iNDAwIiBmaWxsPSIjRjNGNEY2Ii8+CjxwYXRoIGQ9Ik0yMDAgMTUwQzIyNy42MTQgMTUwIDI1MCAxNzIuMzg2IDI1MCAyMDBDMjUwIDIyNy42MTQgMjI3LjYxNCAyNTAgMjAwIDI1MEMxNzIuMzg2IDI1MCAxNTAgMjI3LjYxNCAxNTAgMjAwQzE1MCAxNzIuMzg2IDE3Mi4zODYgMTUwIDIwMCAxNTBaIiBmaWxsPSIjOUNBM0FGIi8+CjxwYXRoIGQ9Ik0yMDAgMTgwQzIxMS4wNDYgMTgwIDIyMCAxODguOTU0IDIyMCAyMDBDMjIwIDIxMS4wNDYgMjExLjA0NiAyMjAgMjAwIDIyMEMxODguOTU0IDIyMCAxODAgMjExLjA0NiAxODAgMjAwQzE4MCAxODguOTU0IDE4OC45NTQgMTgwIDIwMCAxODBaIiBmaWxsPSIjRjNGNEY2Ii8+Cjwvc3ZnPgo=')
    } finally {
      setIsGenerating(false)
    }
  }

  const handleGenerateVideo = async () => {
    if (!videoPrompt.trim()) {
      setError('Please enter a video description')
      return
    }

    setIsGenerating(true)
    setError('')
    
    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'video',
          prompt: videoPrompt,
          options: {
            duration: videoDuration,
            style: videoStyle,
            format: videoFormat
          }
        })
      })

      if (!response.ok) {
        throw new Error('Failed to generate video')
      }

      const data = await response.json()
      
      if (data.error) {
        throw new Error(data.error)
      }

      setGeneratedVideo(data.video || '')
    } catch (error) {
      console.error('Error generating video:', error)
      setError(error instanceof Error ? error.message : 'Failed to generate video')
      // Don't set a placeholder - let the user try again
    } finally {
      setIsGenerating(false)
    }
  }

  const handleSendMessage = async () => {
    if (!chatInput.trim()) return

    const userMessage = { role: 'user' as const, content: chatInput }
    setChatMessages(prev => [...prev, userMessage])
    const input = chatInput
    setChatInput('')

    try {
      const response = await fetch('/api/ai', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          type: 'content',
          prompt: input,
          options: {
            contentType: 'chat response',
            tone: 'helpful',
            length: 'short'
          }
        })
      })

      const data = await response.json()
      const assistantMessage = { 
        role: 'assistant' as const, 
        content: data.content || "I'm here to help! What would you like to know?" 
      }
      setChatMessages(prev => [...prev, assistantMessage])
    } catch (error) {
      const errorMessage = { 
        role: 'assistant' as const, 
        content: "I'm sorry, I encountered an error. Please try again." 
      }
      setChatMessages(prev => [...prev, errorMessage])
    }
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">AI Tools</h1>
          <p className="text-gray-600 mt-2">Leverage AI to boost your social media and SEO performance</p>
        </div>
        <div className="flex items-center space-x-2">
          <Badge variant="secondary" className="bg-green-100 text-green-800">
            <Sparkles className="w-3 h-3 mr-1" />
            AI Powered
          </Badge>
        </div>
      </div>

      {/* AI Tools Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {aiTools.map((tool) => {
          const Icon = tool.icon
          return (
            <Card 
              key={tool.id}
              className={`cursor-pointer transition-all hover:shadow-lg ${
                selectedTool === tool.id ? 'ring-2 ring-purple-600' : ''
              }`}
              onClick={() => setSelectedTool(tool.id)}
            >
              <CardContent className="p-6">
                <div className={`w-12 h-12 bg-gradient-to-r ${tool.color} rounded-lg flex items-center justify-center mb-4`}>
                  <Icon className="h-6 w-6 text-white" />
                </div>
                <h3 className="font-semibold text-gray-900 mb-2">{tool.name}</h3>
                <p className="text-sm text-gray-600">{tool.description}</p>
              </CardContent>
            </Card>
          )
        })}
      </div>

      <Tabs defaultValue="generator" className="space-y-6">
        <TabsList>
          <TabsTrigger value="generator">Content Generator</TabsTrigger>
          <TabsTrigger value="optimizer">Content Optimizer</TabsTrigger>
          <TabsTrigger value="automation">AI Automation</TabsTrigger>
          <TabsTrigger value="analytics">AI Analytics</TabsTrigger>
        </TabsList>

        <TabsContent value="generator">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Dynamic Tool Interface */}
            <Card>
              <CardHeader>
                <CardTitle>
                  {selectedTool === 'content' && 'AI Content Generator'}
                  {selectedTool === 'image' && 'AI Image Generator'}
                  {selectedTool === 'video' && 'AI Video Generator'}
                  {selectedTool === 'chatbot' && 'AI Chatbot Assistant'}
                </CardTitle>
                <CardDescription>
                  {selectedTool === 'content' && 'Create high-quality content with advanced AI'}
                  {selectedTool === 'image' && 'Generate stunning visuals with AI'}
                  {selectedTool === 'video' && 'Generate actual videos from text prompts'}
                  {selectedTool === 'chatbot' && 'Chat with AI assistant for instant help'}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                {error && (
                  <div className="p-3 bg-red-50 border border-red-200 rounded-md">
                    <p className="text-sm text-red-600">{error}</p>
                  </div>
                )}

                {selectedTool === 'content' && (
                  <>
                    <div>
                      <Label className="text-sm font-medium text-gray-700 mb-3 block">Content Type</Label>
                      <div className="grid grid-cols-2 gap-3">
                        {contentTemplates.map((template, index) => (
                          <Button
                            key={index}
                            variant="outline"
                            className={`justify-start h-auto p-3 ${contentType === template.name ? 'bg-purple-50 border-purple-200' : ''}`}
                            onClick={() => setContentType(template.name)}
                          >
                            <div className="text-left">
                              <div className="font-medium text-sm">{template.name}</div>
                              <div className="text-xs text-gray-500">{template.description}</div>
                            </div>
                          </Button>
                        ))}
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="topic" className="text-sm font-medium text-gray-700 mb-2 block">
                        Topic/Keyword
                      </Label>
                      <Input
                        id="topic"
                        placeholder="Enter your topic or primary keyword..."
                        value={topic}
                        onChange={(e) => setTopic(e.target.value)}
                      />
                    </div>

                    <div>
                      <Label htmlFor="tone" className="text-sm font-medium text-gray-700 mb-2 block">
                        Content Tone
                      </Label>
                      <Select value={tone} onValueChange={setTone}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select tone" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="professional">Professional</SelectItem>
                          <SelectItem value="casual">Casual</SelectItem>
                          <SelectItem value="enthusiastic">Enthusiastic</SelectItem>
                          <SelectItem value="persuasive">Persuasive</SelectItem>
                          <SelectItem value="informative">Informative</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="length" className="text-sm font-medium text-gray-700 mb-2 block">
                        Content Length
                      </Label>
                      <Select value={length} onValueChange={setLength}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select length" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="short">Short (100-300 words)</SelectItem>
                          <SelectItem value="medium">Medium (300-600 words)</SelectItem>
                          <SelectItem value="long">Long (600-1000 words)</SelectItem>
                          <SelectItem value="comprehensive">Comprehensive (1000+ words)</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="additional" className="text-sm font-medium text-gray-700 mb-2 block">
                        Additional Instructions
                      </Label>
                      <Textarea
                        id="additional"
                        placeholder="Any specific requirements or guidelines..."
                        rows={3}
                        value={additionalInstructions}
                        onChange={(e) => setAdditionalInstructions(e.target.value)}
                      />
                    </div>

                    <Button 
                      className="w-full" 
                      onClick={handleGenerate}
                      disabled={isGenerating}
                    >
                      {isGenerating ? (
                        <>
                          <Brain className="mr-2 h-4 w-4 animate-pulse" />
                          Generating...
                        </>
                      ) : (
                        <>
                          <Sparkles className="mr-2 h-4 w-4" />
                          Generate Content
                        </>
                      )}
                    </Button>
                  </>
                )}

                {selectedTool === 'image' && (
                  <>
                    <div>
                      <Label htmlFor="image-prompt" className="text-sm font-medium text-gray-700 mb-2 block">
                        Image Description
                      </Label>
                      <Textarea
                        id="image-prompt"
                        placeholder="Describe the image you want to generate..."
                        rows={4}
                        value={imagePrompt}
                        onChange={(e) => setImagePrompt(e.target.value)}
                      />
                    </div>

                    <div>
                      <Label htmlFor="image-style" className="text-sm font-medium text-gray-700 mb-2 block">
                        Image Style
                      </Label>
                      <Select value={imageStyle} onValueChange={setImageStyle}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select style" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="realistic">Realistic</SelectItem>
                          <SelectItem value="artistic">Artistic</SelectItem>
                          <SelectItem value="cartoon">Cartoon</SelectItem>
                          <SelectItem value="abstract">Abstract</SelectItem>
                          <SelectItem value="vintage">Vintage</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <Button 
                      className="w-full" 
                      onClick={handleGenerateImage}
                      disabled={isGenerating}
                    >
                      {isGenerating ? (
                        <>
                          <Brain className="mr-2 h-4 w-4 animate-pulse" />
                          Generating...
                        </>
                      ) : (
                        <>
                          <ImageIcon className="mr-2 h-4 w-4" />
                          Generate Image
                        </>
                      )}
                    </Button>
                  </>
                )}

                {selectedTool === 'video' && (
                  <>
                    <div>
                      <Label htmlFor="video-prompt" className="text-sm font-medium text-gray-700 mb-2 block">
                        Video Description
                      </Label>
                      <Textarea
                        id="video-prompt"
                        placeholder="Describe the video you want to generate..."
                        rows={4}
                        value={videoPrompt}
                        onChange={(e) => setVideoPrompt(e.target.value)}
                      />
                    </div>

                    <div>
                      <Label htmlFor="video-duration" className="text-sm font-medium text-gray-700 mb-2 block">
                        Video Duration
                      </Label>
                      <Select value={videoDuration} onValueChange={setVideoDuration}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select duration" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="5">5 seconds (Quick clip)</SelectItem>
                          <SelectItem value="10">10 seconds (Short/Reel)</SelectItem>
                          <SelectItem value="30">30 seconds (Standard)</SelectItem>
                          <SelectItem value="60">60 seconds (Extended)</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="video-style" className="text-sm font-medium text-gray-700 mb-2 block">
                        Video Style
                      </Label>
                      <Select value={videoStyle} onValueChange={setVideoStyle}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select style" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="realistic">Realistic</SelectItem>
                          <SelectItem value="cinematic">Cinematic</SelectItem>
                          <SelectItem value="anime">Anime</SelectItem>
                          <SelectItem value="cartoon">Cartoon</SelectItem>
                          <SelectItem value="abstract">Abstract</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="video-format" className="text-sm font-medium text-gray-700 mb-2 block">
                        Output Format
                      </Label>
                      <Select value={videoFormat} onValueChange={setVideoFormat}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select format" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="mp4">MP4 (Recommended)</SelectItem>
                          <SelectItem value="webm">WebM</SelectItem>
                          <SelectItem value="mov">MOV</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <Button 
                      className="w-full" 
                      onClick={handleGenerateVideo}
                      disabled={isGenerating}
                    >
                      {isGenerating ? (
                        <>
                          <Brain className="mr-2 h-4 w-4 animate-pulse" />
                          Generating Video...
                        </>
                      ) : (
                        <>
                          <Video className="mr-2 h-4 w-4" />
                          Generate Video
                        </>
                      )}
                    </Button>
                  </>
                )}

                {selectedTool === 'chatbot' && (
                  <div className="h-[600px]">
                    <ModernChatbot />
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Dynamic Results Display */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div>
                    <CardTitle>
                      {selectedTool === 'content' && 'Generated Content'}
                      {selectedTool === 'image' && 'Generated Image'}
                      {selectedTool === 'video' && 'Generated Video'}
                      {selectedTool === 'chatbot' && 'Chat History'}
                    </CardTitle>
                    <CardDescription>
                      {selectedTool === 'content' && 'Preview and edit your AI-generated content'}
                      {selectedTool === 'image' && 'Your AI-generated image will appear here'}
                      {selectedTool === 'video' && 'Preview and download your AI-generated video'}
                      {selectedTool === 'chatbot' && 'View your conversation history'}
                    </CardDescription>
                  </div>
                  {selectedTool === 'content' && (
                    <div className="flex space-x-2">
                      <Button 
                        variant="outline" 
                        size="sm"
                        onClick={handleExport}
                        className="hover:scale-105 transition-transform"
                      >
                        <Download className="h-4 w-4 mr-1" />
                        Export
                      </Button>
                      <Button 
                        variant="outline" 
                        size="sm"
                        onClick={handleShare}
                        className="hover:scale-105 transition-transform"
                      >
                        <Share className="h-4 w-4 mr-1" />
                        Share
                      </Button>
                    </div>
                  )}
                </div>
              </CardHeader>
              <CardContent>
                {selectedTool === 'content' && (
                  <>
                    {generatedContent ? (
                      <div className="space-y-4">
                        <Textarea
                          value={generatedContent}
                          onChange={(e) => setGeneratedContent(e.target.value)}
                          rows={12}
                          className="resize-none"
                        />
                        <div className="flex space-x-3">
                          <Button 
                            variant="outline" 
                            className="flex-1 hover:scale-105 transition-transform"
                            onClick={handleRegenerate}
                            disabled={isGenerating}
                          >
                            <Brain className="mr-2 h-4 w-4" />
                            Regenerate
                          </Button>
                          <Button 
                            className="flex-1 hover:scale-105 transition-transform"
                            onClick={handleOptimizeSEO}
                            disabled={isGenerating}
                          >
                            <Target className="mr-2 h-4 w-4" />
                            Optimize for SEO
                          </Button>
                        </div>
                      </div>
                    ) : (
                      <div className="text-center py-12">
                        <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                        <h3 className="text-lg font-medium text-gray-900 mb-2">No content generated yet</h3>
                        <p className="text-gray-600">Configure your settings and click generate to create content</p>
                      </div>
                    )}
                  </>
                )}

                {selectedTool === 'image' && (
                  <>
                    {generatedImage ? (
                      <div className="space-y-4">
                        <div className="relative">
                          <img 
                            src={generatedImage} 
                            alt="Generated image" 
                            className="w-full rounded-lg shadow-lg"
                          />
                          <Button 
                            variant="outline" 
                            size="sm"
                            className="absolute top-2 right-2 bg-white/90 hover:bg-white"
                            onClick={() => {
                              const link = document.createElement('a')
                              link.href = generatedImage
                              link.download = 'ai-generated-image.png'
                              link.click()
                            }}
                          >
                            <Download className="h-4 w-4 mr-1" />
                            Download
                          </Button>
                        </div>
                        <Button 
                          variant="outline" 
                          className="w-full hover:scale-105 transition-transform"
                          onClick={handleGenerateImage}
                          disabled={isGenerating}
                        >
                          <Brain className="mr-2 h-4 w-4" />
                          Generate New Image
                        </Button>
                      </div>
                    ) : (
                      <div className="text-center py-12">
                        <ImageIcon className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                        <h3 className="text-lg font-medium text-gray-900 mb-2">No image generated yet</h3>
                        <p className="text-gray-600">Describe the image you want to generate and click generate</p>
                      </div>
                    )}
                  </>
                )}

                {selectedTool === 'video' && (
                  <>
                    {generatedVideo ? (
                      <div className="space-y-4">
                        <div className="relative bg-black rounded-lg overflow-hidden" style={{ height: '500px' }}>
                          {generatedVideo.startsWith('blob:') ? (
                            // Display HTML video in iframe
                            <iframe 
                              src={generatedVideo}
                              className="w-full h-full rounded-lg border-0"
                              title="AI Generated Video"
                              allow="autoplay; fullscreen"
                            />
                          ) : generatedVideo.startsWith('data:image/') ? (
                            // Display generated image as video preview
                            <div className="relative w-full h-full">
                              <img 
                                src={generatedVideo}
                                alt="Generated video preview"
                                className="w-full h-full object-cover"
                              />
                              <div className="absolute inset-0 bg-black bg-opacity-40 flex items-center justify-center">
                                <div className="text-center text-white">
                                  <Video className="h-16 w-16 mx-auto mb-4 animate-pulse" />
                                  <h3 className="text-xl font-bold mb-2">AI Video Generated</h3>
                                  <p className="text-sm opacity-90">Your video: "{videoPrompt}"</p>
                                  <div className="mt-4 flex items-center justify-center space-x-4">
                                    <Badge className="bg-green-500">Duration: {videoDuration}s</Badge>
                                    <Badge className="bg-blue-500">Style: {videoStyle}</Badge>
                                    <Badge className="bg-purple-500">Format: {videoFormat}</Badge>
                                  </div>
                                </div>
                              </div>
                              <div className="absolute bottom-4 left-4 right-4">
                                <div className="bg-white bg-opacity-90 rounded-lg p-3">
                                  <div className="flex items-center space-x-2">
                                    <div className="animate-pulse bg-green-500 h-2 w-2 rounded-full"></div>
                                    <span className="text-sm font-medium">Video Ready</span>
                                  </div>
                                </div>
                              </div>
                            </div>
                          ) : (
                            // Display actual video if available
                            <video 
                              controls 
                              className="w-full h-full"
                              src={generatedVideo}
                            >
                              Your browser does not support the video tag.
                            </video>
                          )}
                          
                          <Button 
                            variant="outline" 
                            size="sm"
                            className="absolute top-2 right-2 bg-white/90 hover:bg-white z-10"
                            onClick={() => {
                              if (generatedVideo.startsWith('blob:')) {
                                // For HTML videos, open in new tab
                                window.open(generatedVideo, '_blank')
                              } else if (generatedVideo.startsWith('data:image/')) {
                                const link = document.createElement('a')
                                link.href = generatedVideo
                                link.download = `ai-generated-video-${Date.now()}.png`
                                link.click()
                              } else {
                                const link = document.createElement('a')
                                link.href = generatedVideo
                                link.download = `ai-generated-video-${Date.now()}.${videoFormat}`
                                link.click()
                              }
                            }}
                          >
                            <Download className="h-4 w-4 mr-1" />
                            {generatedVideo.startsWith('blob:') ? 'Open' : 'Download'}
                          </Button>
                        </div>
                        
                        <div className="grid grid-cols-2 gap-3">
                          <Button 
                            variant="outline" 
                            className="hover:scale-105 transition-transform"
                            onClick={handleGenerateVideo}
                            disabled={isGenerating}
                          >
                            <Brain className="mr-2 h-4 w-4" />
                            Generate New
                          </Button>
                          <Button 
                            variant="outline" 
                            className="hover:scale-105 transition-transform"
                            onClick={() => setGeneratedVideo('')}
                          >
                            Clear
                          </Button>
                        </div>
                        
                        {/* Video Generation Info */}
                        <div className="bg-gray-50 rounded-lg p-4">
                          <h4 className="font-medium text-gray-900 mb-2">Generation Details</h4>
                          <div className="text-sm text-gray-600 space-y-1">
                            <p><strong>Prompt:</strong> {videoPrompt}</p>
                            <p><strong>Duration:</strong> {videoDuration} seconds</p>
                            <p><strong>Style:</strong> {videoStyle}</p>
                            <p><strong>Format:</strong> {videoFormat.toUpperCase()}</p>
                            <p><strong>Generated:</strong> {new Date().toLocaleString()}</p>
                            <p><strong>Type:</strong> {generatedVideo.startsWith('blob:') ? 'Interactive HTML Video' : 'Static Preview'}</p>
                          </div>
                        </div>
                      </div>
                    ) : (
                      <div className="text-center py-12">
                        <Video className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                        <h3 className="text-lg font-medium text-gray-900 mb-2">No video generated yet</h3>
                        <p className="text-gray-600">Describe the video you want to generate and click generate</p>
                        <div className="mt-6 flex flex-wrap justify-center gap-2">
                          <Badge variant="outline" className="border-green-200 text-green-700">
                            ✨ AI Powered
                          </Badge>
                          <Badge variant="outline" className="border-blue-200 text-blue-700">
                            🎬 Playable Video
                          </Badge>
                          <Badge variant="outline" className="border-purple-200 text-purple-700">
                            🎨 Multiple Styles
                          </Badge>
                          <Badge variant="outline" className="border-orange-200 text-orange-700">
                            🌐 Interactive HTML
                          </Badge>
                        </div>
                      </div>
                    )}
                  </>
                )}

                {selectedTool === 'chatbot' && (
                  <div className="text-center py-12">
                    <div className="w-16 h-16 bg-gradient-to-r from-blue-600 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
                      <MessageSquare className="h-8 w-8 text-white" />
                    </div>
                    <h3 className="text-xl font-bold text-gray-900 mb-2">Modern AI Chatbot</h3>
                    <p className="text-gray-600 mb-4">Experience the future of AI conversations</p>
                    <div className="flex flex-wrap justify-center gap-2">
                      <Badge className="bg-gradient-to-r from-purple-500 to-pink-500 text-white border-0">
                        <Sparkles className="w-3 h-3 mr-1" />
                        AI Powered
                      </Badge>
                      <Badge variant="outline" className="border-purple-200 text-purple-700">
                        Modern UI
                      </Badge>
                      <Badge variant="outline" className="border-blue-200 text-blue-700">
                        Real-time
                      </Badge>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="optimizer">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Content Optimizer</CardTitle>
                <CardDescription>Optimize existing content for better performance</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="content-url" className="text-sm font-medium text-gray-700 mb-2 block">
                    Content URL or Paste Content
                  </Label>
                  <Textarea
                    id="content-url"
                    placeholder="Paste your content here or enter a URL..."
                    rows={6}
                  />
                </div>
                
                <div>
                  <Label htmlFor="target-keywords" className="text-sm font-medium text-gray-700 mb-2 block">
                    Target Keywords
                  </Label>
                  <Input
                    id="target-keywords"
                    placeholder="Enter target keywords (comma separated)..."
                  />
                </div>

                <Button 
                  className="w-full hover:scale-105 transition-transform"
                  onClick={handleAnalyzeOptimize}
                >
                  <TrendingUp className="mr-2 h-4 w-4" />
                  Analyze & Optimize
                </Button>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Optimization Results</CardTitle>
                <CardDescription>AI-powered suggestions for improvement</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="text-center py-8">
                  <Search className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">Ready to optimize</h3>
                  <p className="text-gray-600">Paste your content to get AI-powered optimization suggestions</p>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="automation">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Social Media Automation</CardTitle>
                <CardDescription>Automate your social media posting</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center space-x-3">
                  <Zap className="h-5 w-5 text-yellow-500" />
                  <span className="text-sm">Auto-post scheduling</span>
                </div>
                <div className="flex items-center space-x-3">
                  <Users className="h-5 w-5 text-blue-500" />
                  <span className="text-sm">Audience targeting</span>
                </div>
                <div className="flex items-center space-x-3">
                  <TrendingUp className="h-5 w-5 text-green-500" />
                  <span className="text-sm">Performance optimization</span>
                </div>
                <Button 
                  className="w-full mt-4 hover:scale-105 transition-transform"
                  onClick={() => handleConfigureAutomation('Social Media')}
                >
                  Configure Automation
                </Button>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>SEO Automation</CardTitle>
                <CardDescription>Automated SEO optimization</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center space-x-3">
                  <Search className="h-5 w-5 text-purple-500" />
                  <span className="text-sm">Keyword tracking</span>
                </div>
                <div className="flex items-center space-x-3">
                  <Globe className="h-5 w-5 text-blue-500" />
                  <span className="text-sm">Rank monitoring</span>
                </div>
                <div className="flex items-center space-x-3">
                  <Target className="h-5 w-5 text-red-500" />
                  <span className="text-sm">Competitor analysis</span>
                </div>
                <Button 
                  className="w-full mt-4 hover:scale-105 transition-transform"
                  onClick={() => handleConfigureAutomation('SEO')}
                >
                  Setup SEO Automation
                </Button>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Content Automation</CardTitle>
                <CardDescription>Automated content creation</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center space-x-3">
                  <FileText className="h-5 w-5 text-green-500" />
                  <span className="text-sm">Blog generation</span>
                </div>
                <div className="flex items-center space-x-3">
                  <MessageSquare className="h-5 w-5 text-blue-500" />
                  <span className="text-sm">Social media posts</span>
                </div>
                <div className="flex items-center space-x-3">
                  <Video className="h-5 w-5 text-purple-500" />
                  <span className="text-sm">Video scripts</span>
                </div>
                <Button 
                  className="w-full mt-4 hover:scale-105 transition-transform"
                  onClick={() => handleConfigureAutomation('Content')}
                >
                  Create Automation
                </Button>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="analytics">
          <Card>
            <CardHeader>
              <CardTitle>AI Analytics Dashboard</CardTitle>
              <CardDescription>Advanced insights powered by artificial intelligence</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <div className="text-center">
                  <div className="text-2xl font-bold text-purple-600">94%</div>
                  <p className="text-sm text-gray-600">Content Accuracy</p>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-blue-600">87%</div>
                  <p className="text-sm text-gray-600">Engagement Prediction</p>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-green-600">92%</div>
                  <p className="text-sm text-gray-600">SEO Score</p>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-orange-600">78%</div>
                  <p className="text-sm text-gray-600">Conversion Rate</p>
                </div>
              </div>
              
              <div className="mt-8 text-center">
                <Brain className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">AI-Powered Insights</h3>
                <p className="text-gray-600">Get predictive analytics and recommendations to boost your performance</p>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
}