import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const accountId = searchParams.get('accountId')
    const status = searchParams.get('status')
    
    const socialPosts = await db.socialPost.findMany({
      where: {
        ...(accountId && { accountId }),
        ...(status && { status })
      },
      include: {
        account: true,
        analytics: true
      },
      orderBy: {
        createdAt: 'desc'
      }
    })
    
    return NextResponse.json(socialPosts)
  } catch (error) {
    console.error('Error fetching social posts:', error)
    return NextResponse.json({ error: 'Failed to fetch social posts' }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const { 
      platform, 
      content, 
      mediaUrls, 
      hashtags, 
      scheduledFor, 
      accountId 
    } = await request.json()
    
    const socialPost = await db.socialPost.create({
      data: {
        platform,
        content,
        mediaUrls: mediaUrls ? JSON.stringify(mediaUrls) : null,
        hashtags: hashtags ? JSON.stringify(hashtags) : null,
        scheduledFor: scheduledFor ? new Date(scheduledFor) : null,
        accountId,
        status: 'draft'
      }
    })
    
    return NextResponse.json(socialPost, { status: 201 })
  } catch (error) {
    console.error('Error creating social post:', error)
    return NextResponse.json({ error: 'Failed to create social post' }, { status: 500 })
  }
}