import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const domainId = searchParams.get('domainId')
    const keywordId = searchParams.get('keywordId')
    const country = searchParams.get('country')
    
    const rankings = await db.ranking.findMany({
      where: {
        ...(domainId && { domainId }),
        ...(keywordId && { keywordId }),
        ...(country && { country })
      },
      include: {
        keyword: true,
        domain: true
      },
      orderBy: {
        trackedAt: 'desc'
      }
    })
    
    return NextResponse.json(rankings)
  } catch (error) {
    console.error('Error fetching rankings:', error)
    return NextResponse.json({ error: 'Failed to fetch rankings' }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const { 
      keywordId, 
      domainId, 
      position, 
      url, 
      country, 
      state, 
      city, 
      searchEngine 
    } = await request.json()
    
    const ranking = await db.ranking.create({
      data: {
        keywordId,
        domainId,
        position,
        url,
        country,
        state,
        city,
        searchEngine
      }
    })
    
    return NextResponse.json(ranking, { status: 201 })
  } catch (error) {
    console.error('Error creating ranking:', error)
    return NextResponse.json({ error: 'Failed to create ranking' }, { status: 500 })
  }
}