import { NextRequest, NextResponse } from 'next/server'

interface LocationData {
  id: string
  lat: number
  lng: number
  city: string
  country: string
  views: number
  ranking: number
  platform: string
  trend: 'up' | 'down' | 'stable'
  growth: number
  timestamp: string
}

// Mock data - in a real app, this would come from a database
const mockLocationData: LocationData[] = [
  {
    id: '1',
    lat: 40.7128,
    lng: -74.0060,
    city: 'New York',
    country: 'USA',
    views: 15420,
    ranking: 3,
    platform: 'Instagram',
    trend: 'up',
    growth: 12.5,
    timestamp: new Date().toISOString()
  },
  {
    id: '2',
    lat: 34.0522,
    lng: -118.2437,
    city: 'Los Angeles',
    country: 'USA',
    views: 12350,
    ranking: 5,
    platform: 'TikTok',
    trend: 'up',
    growth: 8.3,
    timestamp: new Date().toISOString()
  },
  {
    id: '3',
    lat: 51.5074,
    lng: -0.1278,
    city: 'London',
    country: 'UK',
    views: 9800,
    ranking: 8,
    platform: 'YouTube',
    trend: 'stable',
    growth: 2.1,
    timestamp: new Date().toISOString()
  },
  {
    id: '4',
    lat: 35.6762,
    lng: 139.6503,
    city: 'Tokyo',
    country: 'Japan',
    views: 8900,
    ranking: 12,
    platform: 'Twitter',
    trend: 'down',
    growth: -3.2,
    timestamp: new Date().toISOString()
  },
  {
    id: '5',
    lat: 48.8566,
    lng: 2.3522,
    city: 'Paris',
    country: 'France',
    views: 7600,
    ranking: 15,
    platform: 'Instagram',
    trend: 'up',
    growth: 6.7,
    timestamp: new Date().toISOString()
  },
  {
    id: '6',
    lat: 52.5200,
    lng: 13.4050,
    city: 'Berlin',
    country: 'Germany',
    views: 6500,
    ranking: 18,
    platform: 'LinkedIn',
    trend: 'up',
    growth: 4.2,
    timestamp: new Date().toISOString()
  },
  {
    id: '7',
    lat: 43.6532,
    lng: -79.3832,
    city: 'Toronto',
    country: 'Canada',
    views: 5800,
    ranking: 22,
    platform: 'Instagram',
    trend: 'stable',
    growth: 1.8,
    timestamp: new Date().toISOString()
  },
  {
    id: '8',
    lat: -33.8688,
    lng: 151.2093,
    city: 'Sydney',
    country: 'Australia',
    views: 5200,
    ranking: 25,
    platform: 'TikTok',
    trend: 'up',
    growth: 9.1,
    timestamp: new Date().toISOString()
  }
]

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const metric = searchParams.get('metric') || 'views'
    const limit = searchParams.get('limit')
    
    let filteredData = [...mockLocationData]
    
    // Sort by the requested metric
    switch (metric) {
      case 'views':
        filteredData.sort((a, b) => b.views - a.views)
        break
      case 'ranking':
        filteredData.sort((a, b) => a.ranking - b.ranking)
        break
      case 'growth':
        filteredData.sort((a, b) => b.growth - a.growth)
        break
    }
    
    // Apply limit if specified
    if (limit) {
      filteredData = filteredData.slice(0, parseInt(limit))
    }
    
    // Simulate real-time data updates
    const updatedData = filteredData.map(location => ({
      ...location,
      views: location.views + Math.floor(Math.random() * 100),
      growth: location.growth + (Math.random() - 0.5) * 2,
      timestamp: new Date().toISOString()
    }))
    
    return NextResponse.json({
      success: true,
      data: updatedData,
      total: updatedData.length,
      timestamp: new Date().toISOString()
    })
  } catch (error) {
    console.error('Error fetching map data:', error)
    return NextResponse.json(
      { success: false, error: 'Failed to fetch map data' },
      { status: 500 }
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { action, locationId, data } = body
    
    // In a real app, this would update the database
    // For now, we'll just return a success response
    
    switch (action) {
      case 'update_location':
        console.log(`Updating location ${locationId} with data:`, data)
        break
      case 'add_location':
        console.log('Adding new location:', data)
        break
      default:
        throw new Error('Invalid action')
    }
    
    return NextResponse.json({
      success: true,
      message: 'Location data updated successfully',
      timestamp: new Date().toISOString()
    })
  } catch (error) {
    console.error('Error updating map data:', error)
    return NextResponse.json(
      { success: false, error: 'Failed to update map data' },
      { status: 500 }
    )
  }
}