import { NextRequest, NextResponse } from 'next/server'

interface LiveFeedItem {
  id: string
  type: 'new_view' | 'ranking_change' | 'trend_spike' | 'viral_content' | 'engagement_surge'
  location: string
  message: string
  timestamp: string
  importance: 'high' | 'medium' | 'low'
  data?: {
    views?: number
    ranking?: number
    growth?: number
    platform?: string
  }
}

// Mock live feed data
const generateMockFeedItem = (): LiveFeedItem => {
  const cities = [
    'New York, USA',
    'Los Angeles, USA',
    'London, UK',
    'Tokyo, Japan',
    'Paris, France',
    'Berlin, Germany',
    'Toronto, Canada',
    'Sydney, Australia',
    'Mumbai, India',
    'São Paulo, Brazil'
  ]
  
  const platforms = ['Instagram', 'TikTok', 'YouTube', 'Twitter', 'LinkedIn']
  const types: LiveFeedItem['type'][] = ['new_view', 'ranking_change', 'trend_spike', 'viral_content', 'engagement_surge']
  
  const type = types[Math.floor(Math.random() * types.length)]
  const city = cities[Math.floor(Math.random() * cities.length)]
  const platform = platforms[Math.floor(Math.random() * platforms.length)]
  
  const messages = {
    new_view: [
      `Surge in views: +${Math.floor(Math.random() * 1000)} in last 5 minutes`,
      `Rapid view increase detected on ${platform}`,
      `Unusual view activity: ${Math.floor(Math.random() * 500)} new views`
    ],
    ranking_change: [
      `Ranking improved: #${Math.floor(Math.random() * 20) + 1} → #${Math.floor(Math.random() * 20) + 1} on ${platform}`,
      `Climbed ${Math.floor(Math.random() * 10) + 1} positions in ${platform} rankings`,
      `New ranking milestone: Top ${Math.floor(Math.random() * 50) + 1} on ${platform}`
    ],
    trend_spike: [
      `Engagement spike detected: ${Math.floor(Math.random() * 300) + 100}% increase`,
      `Viral trend detected in ${city}`,
      `Sudden popularity surge on ${platform}`
    ],
    viral_content: [
      `Content going viral in ${city}`,
      `Viral potential detected: ${Math.floor(Math.random() * 10000)} impressions`,
      `Exponential growth pattern identified`
    ],
    engagement_surge: [
      `Engagement rate: ${(Math.random() * 20 + 5).toFixed(1)}% (above average)`,
      `High engagement activity from ${city}`,
      `Comment and share surge detected`
    ]
  }
  
  const typeMessages = messages[type]
  const message = typeMessages[Math.floor(Math.random() * typeMessages.length)]
  
  return {
    id: Date.now().toString() + Math.random().toString(36).substr(2, 9),
    type,
    location: city,
    message,
    timestamp: new Date().toISOString(),
    importance: Math.random() > 0.8 ? 'high' : Math.random() > 0.5 ? 'medium' : 'low',
    data: {
      views: type === 'new_view' ? Math.floor(Math.random() * 1000) : undefined,
      ranking: type === 'ranking_change' ? Math.floor(Math.random() * 50) + 1 : undefined,
      growth: Math.floor(Math.random() * 100) - 50,
      platform
    }
  }
}

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const limit = searchParams.get('limit') || '10'
    const type = searchParams.get('type')
    
    // Generate mock feed items
    let feedItems: LiveFeedItem[] = []
    const itemCount = Math.min(parseInt(limit), 50) // Cap at 50 items
    
    for (let i = 0; i < itemCount; i++) {
      const item = generateMockFeedItem()
      // Stagger timestamps to make it look more realistic
      item.timestamp = new Date(Date.now() - i * Math.random() * 300000).toISOString()
      feedItems.push(item)
    }
    
    // Filter by type if specified
    if (type) {
      feedItems = feedItems.filter(item => item.type === type)
    }
    
    // Sort by timestamp (newest first)
    feedItems.sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime())
    
    return NextResponse.json({
      success: true,
      data: feedItems,
      total: feedItems.length,
      timestamp: new Date().toISOString()
    })
  } catch (error) {
    console.error('Error fetching live feed:', error)
    return NextResponse.json(
      { success: false, error: 'Failed to fetch live feed data' },
      { status: 500 }
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { action, data } = body
    
    switch (action) {
      case 'add_event':
        // In a real app, this would add to a database or event stream
        console.log('Adding new live event:', data)
        break
      case 'mark_read':
        console.log('Marking event as read:', data)
        break
      default:
        throw new Error('Invalid action')
    }
    
    return NextResponse.json({
      success: true,
      message: 'Live feed action completed successfully',
      timestamp: new Date().toISOString()
    })
  } catch (error) {
    console.error('Error processing live feed action:', error)
    return NextResponse.json(
      { success: false, error: 'Failed to process live feed action' },
      { status: 500 }
    )
  }
}