import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import ZAI from 'z-ai-web-dev-sdk'

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const userId = searchParams.get('userId')
    const domainId = searchParams.get('domainId')
    const type = searchParams.get('type')
    
    const contents = await db.content.findMany({
      where: {
        ...(userId && { userId }),
        ...(domainId && { domainId }),
        ...(type && { type })
      },
      include: {
        user: true,
        domain: true
      },
      orderBy: {
        createdAt: 'desc'
      }
    })
    
    return NextResponse.json(contents)
  } catch (error) {
    console.error('Error fetching content:', error)
    return NextResponse.json({ error: 'Failed to fetch content' }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const { title, content, type, domainId, userId, keywords, aiGenerated } = await request.json()
    
    const contentRecord = await db.content.create({
      data: {
        title,
        content,
        type,
        domainId,
        userId,
        keywords: keywords ? JSON.stringify(keywords) : null,
        aiGenerated: aiGenerated || false,
        status: 'draft'
      }
    })
    
    return NextResponse.json(contentRecord, { status: 201 })
  } catch (error) {
    console.error('Error creating content:', error)
    return NextResponse.json({ error: 'Failed to create content' }, { status: 500 })
  }
}

export async function PUT(request: NextRequest) {
  try {
    const { id, prompt, contentType, tone, length } = await request.json()
    
    // Generate content using AI
    const zai = await ZAI.create()
    
    const aiPrompt = `Generate a ${contentType} about "${prompt}" with a ${tone} tone. The content should be approximately ${length} words and optimized for SEO and engagement.`
    
    const completion = await zai.chat.completions.create({
      messages: [
        {
          role: 'system',
          content: 'You are a professional content writer and SEO expert. Create high-quality, engaging content that performs well on social media and search engines.'
        },
        {
          role: 'user',
          content: aiPrompt
        }
      ],
      temperature: 0.7,
      max_tokens: 2000
    })
    
    const generatedContent = completion.choices[0]?.message?.content
    
    if (!generatedContent) {
      return NextResponse.json({ error: 'Failed to generate content' }, { status: 500 })
    }
    
    // Update the content with AI-generated text
    const updatedContent = await db.content.update({
      where: { id },
      data: {
        content: generatedContent,
        aiGenerated: true,
        status: 'generated'
      }
    })
    
    return NextResponse.json(updatedContent)
  } catch (error) {
    console.error('Error generating content:', error)
    return NextResponse.json({ error: 'Failed to generate content' }, { status: 500 })
  }
}