import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const userId = searchParams.get('userId')
    const accountId = searchParams.get('accountId')
    const dateRange = searchParams.get('dateRange')
    
    // Calculate date range
    const endDate = new Date()
    const startDate = new Date()
    
    switch (dateRange) {
      case '7d':
        startDate.setDate(startDate.getDate() - 7)
        break
      case '30d':
        startDate.setDate(startDate.getDate() - 30)
        break
      case '90d':
        startDate.setDate(startDate.getDate() - 90)
        break
      default:
        startDate.setDate(startDate.getDate() - 30)
    }
    
    // Get social analytics
    const socialAnalytics = await db.socialAnalytics.findMany({
      where: {
        date: {
          gte: startDate,
          lte: endDate
        },
        ...(accountId && { accountId })
      },
      include: {
        account: true,
        post: true
      },
      orderBy: {
        date: 'asc'
      }
    })
    
    // Get rankings data
    const rankings = await db.ranking.findMany({
      where: {
        trackedAt: {
          gte: startDate,
          lte: endDate
        }
      },
      include: {
        keyword: true,
        domain: true
      }
    })
    
    // Calculate aggregate stats
    const totalFollowers = socialAnalytics.reduce((sum, analytics) => sum + analytics.followers, 0)
    const totalEngagement = socialAnalytics.reduce((sum, analytics) => sum + analytics.likes + analytics.comments + analytics.shares, 0)
    const totalReach = socialAnalytics.reduce((sum, analytics) => sum + analytics.reach, 0)
    const avgPosition = rankings.length > 0 ? rankings.reduce((sum, ranking) => sum + ranking.position, 0) / rankings.length : 0
    
    // Get platform breakdown
    const platformBreakdown = socialAnalytics.reduce((acc, analytics) => {
      const platform = analytics.account?.platform || 'unknown'
      if (!acc[platform]) {
        acc[platform] = {
          followers: 0,
          engagement: 0,
          reach: 0
        }
      }
      acc[platform].followers += analytics.followers
      acc[platform].engagement += analytics.likes + analytics.comments + analytics.shares
      acc[platform].reach += analytics.reach
      return acc
    }, {} as Record<string, { followers: number; engagement: number; reach: number }>)
    
    const analyticsData = {
      overview: {
        totalFollowers,
        totalEngagement,
        totalReach,
        avgPosition: Math.round(avgPosition * 10) / 10,
        totalPosts: socialAnalytics.length,
        totalKeywords: rankings.length
      },
      platformBreakdown,
      socialAnalytics,
      rankings,
      dateRange: {
        startDate,
        endDate
      }
    }
    
    return NextResponse.json(analyticsData)
  } catch (error) {
    console.error('Error fetching analytics:', error)
    return NextResponse.json({ error: 'Failed to fetch analytics' }, { status: 500 })
  }
}