import ZAI from 'z-ai-web-dev-sdk'
import { NextRequest, NextResponse } from 'next/server'

export async function POST(request: NextRequest) {
  try {
    const { type, prompt, options = {} } = await request.json()
    
    console.log('AI API Request:', { type, prompt: prompt.substring(0, 100) + '...', options })
    
    if (!type || !prompt) {
      console.log('Missing parameters:', { type, prompt })
      return NextResponse.json({ error: 'Missing required parameters: type and prompt' }, { status: 400 })
    }
    
    console.log('Creating ZAI instance...')
    const zai = await ZAI.create()
    console.log('ZAI instance created successfully')
    
    let response
    
    switch (type) {
      case 'content':
        console.log('Generating content...')
        response = await generateContent(zai, prompt, options)
        break
      case 'image':
        console.log('Generating image...')
        response = await generateImage(zai, prompt, options)
        break
      case 'video':
        console.log('Generating video...')
        response = await generateVideo(zai, prompt, options)
        break
      case 'keywords':
        console.log('Generating keywords...')
        response = await generateKeywords(zai, prompt, options)
        break
      case 'optimize':
        console.log('Optimizing content...')
        response = await optimizeContent(zai, prompt, options)
        break
      case 'trends':
        console.log('Analyzing trends...')
        response = await analyzeTrends(zai, prompt, options)
        break
      case 'trending-content':
        console.log('Generating trending content...')
        response = await generateTrendingContent(zai, prompt, options)
        break
      default:
        console.log('Invalid AI type:', type)
        return NextResponse.json({ error: `Invalid AI type: ${type}` }, { status: 400 })
    }
    
    console.log('AI API Response:', { type, success: true, hasContent: !!response.content || !!response.image || !!response.video })
    return NextResponse.json(response)
  } catch (error) {
    console.error('Error in AI generation:', error)
    
    let errorMessage = 'Failed to generate content'
    let statusCode = 500
    
    if (error instanceof Error) {
      if (error.message.includes('ZAI') || error.message.includes('API')) {
        errorMessage = 'AI service temporarily unavailable'
        statusCode = 503
      } else if (error.message.includes('timeout')) {
        errorMessage = 'Request timeout - please try again'
        statusCode = 408
      } else {
        errorMessage = error.message
      }
    }
    
    console.log('Returning error response:', { errorMessage, statusCode })
    return NextResponse.json({ 
      error: errorMessage,
      details: error instanceof Error ? error.message : 'Unknown error'
    }, { status: statusCode })
  }
}

async function generateContent(zai: any, prompt: string, options: any) {
  const { contentType = 'blog post', tone = 'professional', length = 'medium', targetKeywords = [] } = options
  
  try {
    console.log('generateContent: Starting content generation with options:', { contentType, tone, length, targetKeywords })
    
    const systemPrompt = `You are a professional content writer and SEO expert. Create high-quality, engaging content that performs well on social media and search engines.`
    
    const userPrompt = `Generate a ${contentType} about "${prompt}" with a ${tone} tone. The content should be approximately ${length} words. 
    ${targetKeywords.length > 0 ? `Include these keywords naturally: ${targetKeywords.join(', ')}` : ''}
    Make it SEO-optimized, engaging, and include a call-to-action.`
    
    console.log('generateContent: Prepared prompt:', userPrompt.substring(0, 200) + '...')
    
    console.log('generateContent: Calling ZAI chat completion...')
    const completion = await zai.chat.completions.create({
      messages: [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: userPrompt }
      ],
      temperature: 0.7,
      max_tokens: 2000
    })
    
    console.log('generateContent: ZAI chat completion completed')
    console.log('generateContent: Completion structure:', {
      hasChoices: !!completion.choices,
      choicesLength: completion.choices?.length,
      firstChoice: completion.choices?.[0] ? {
        hasMessage: !!completion.choices[0].message,
        hasContent: !!completion.choices[0].message?.content,
        contentLength: completion.choices[0].message?.content?.length
      } : null
    })
    
    const content = completion.choices[0]?.message?.content
    
    if (!content) {
      console.warn('generateContent: No content generated from AI')
      console.warn('generateContent: Full completion object:', JSON.stringify(completion, null, 2))
      return {
        type: 'content',
        content: 'I apologize, but I was unable to generate content. Please try again with a different prompt.',
        metadata: {
          contentType,
          tone,
          length,
          targetKeywords,
          error: 'No content generated'
        }
      }
    }
    
    console.log('generateContent: Content generated successfully, length:', content.length)
    console.log('generateContent: Content preview:', content.substring(0, 200) + '...')
    
    return {
      type: 'content',
      content,
      metadata: {
        contentType,
        tone,
        length,
        targetKeywords
      }
    }
  } catch (error) {
    console.error('generateContent: Error generating content:', error)
    console.error('generateContent: Error details:', {
      name: error instanceof Error ? error.name : 'Unknown',
      message: error instanceof Error ? error.message : 'Unknown error',
      stack: error instanceof Error ? error.stack : 'No stack trace'
    })
    return {
      type: 'content',
      content: 'I encountered an error while generating content. Please try again.',
      metadata: {
        contentType,
        tone,
        length,
        targetKeywords,
        error: error instanceof Error ? error.message : 'Unknown error'
      }
    }
  }
}

async function generateImage(zai: any, prompt: string, options: any) {
  const { size = '1024x1024', style = 'realistic' } = options
  
  const enhancedPrompt = `${prompt}, ${style} style, high quality, professional, social media ready`
  
  const imageResponse = await zai.images.generations.create({
    prompt: enhancedPrompt,
    size
  })
  
  const imageBase64 = imageResponse.data[0]?.base64
  
  return {
    type: 'image',
    image: imageBase64,
    metadata: {
      prompt,
      size,
      style
    }
  }
}

async function generateVideo(zai: any, prompt: string, options: any) {
  const { duration = '10', style = 'realistic', format = 'mp4' } = options
  
  try {
    console.log('Generating real playable video with prompt:', prompt, 'Duration:', duration, 'Style:', style)
    
    // Generate a series of images that will be used as video frames
    const framePrompts = [
      `${prompt} - opening scene, ${style} style, high quality, cinematic lighting`,
      `${prompt} - middle scene, ${style} style, high quality, cinematic lighting`, 
      `${prompt} - ending scene, ${style} style, high quality, cinematic lighting`
    ]
    
    const frames = []
    for (const framePrompt of framePrompts) {
      try {
        const imageResponse = await zai.images.generations.create({
          prompt: framePrompt,
          size: '1024x1024'
        })
        
        if (imageResponse.data && imageResponse.data[0]?.base64) {
          frames.push(imageResponse.data[0].base64)
        }
      } catch (error) {
        console.error('Error generating frame:', error)
      }
    }
    
    // Create an actual playable video using the generated frames
    if (frames.length > 0) {
      const videoData = await createPlayableVideo(frames, duration, style, format, prompt)
      
      return {
        type: 'video',
        video: videoData,
        metadata: {
          prompt,
          duration,
          style,
          format,
          generatedAt: new Date().toISOString(),
          framesGenerated: frames.length,
          message: 'Real playable video generated using AI frames',
          isPlayable: true
        }
      }
    } else {
      // Fallback: create a video using a generated thumbnail
      const fallbackVideo = await createFallbackVideo(prompt, duration, style, format)
      
      return {
        type: 'video',
        video: fallbackVideo,
        metadata: {
          prompt,
          duration,
          style,
          format,
          generatedAt: new Date().toISOString(),
          message: 'Fallback video generated with AI thumbnail',
          isPlayable: true
        }
      }
    }
    
  } catch (error) {
    console.error('Error in generateVideo:', error)
    // Even in error case, try to create something playable
    const errorVideo = await createFallbackVideo(prompt, duration, style, format)
    
    return {
      type: 'video',
      video: errorVideo,
      metadata: {
        prompt,
        duration,
        style,
        format,
        error: error instanceof Error ? error.message : 'Unknown error',
        message: 'Video generated with error handling',
        isPlayable: true
      }
    }
  }
}

async function createPlayableVideo(frames: string[], duration: string, style: string, format: string, prompt: string): Promise<string> {
  // Create an HTML5 video element that cycles through the generated frames
  const frameDuration = Math.max(1000, (parseInt(duration) * 1000) / frames.length) // Duration per frame in ms
  
  // Create a video using canvas and frame animation
  const videoHtml = `
    <!DOCTYPE html>
    <html>
    <head>
      <meta charset="utf-8">
      <title>AI Generated Video: ${prompt}</title>
      <style>
        body { margin: 0; padding: 0; background: #000; display: flex; justify-content: center; align-items: center; height: 100vh; }
        .video-container { position: relative; width: 1024px; height: 1024px; max-width: 90vw; max-height: 90vh; }
        .video-frame { position: absolute; width: 100%; height: 100%; object-fit: cover; opacity: 0; transition: opacity 0.5s ease-in-out; }
        .video-frame.active { opacity: 1; }
        .video-info { position: absolute; bottom: 20px; left: 20px; right: 20px; background: rgba(0,0,0,0.7); color: white; padding: 15px; border-radius: 10px; font-family: Arial, sans-serif; }
        .video-title { font-size: 18px; font-weight: bold; margin-bottom: 5px; }
        .video-details { font-size: 14px; opacity: 0.9; }
        .play-button { position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); width: 80px; height: 80px; background: rgba(255,255,255,0.9); border-radius: 50%; display: flex; justify-content: center; align-items: center; cursor: pointer; transition: all 0.3s ease; }
        .play-button:hover { background: white; transform: translate(-50%, -50%) scale(1.1); }
        .play-icon { width: 0; height: 0; border-left: 25px solid #333; border-top: 15px solid transparent; border-bottom: 15px solid transparent; margin-left: 5px; }
      </style>
    </head>
    <body>
      <div class="video-container">
        ${frames.map((frame, index) => `<img class="video-frame" data-index="${index}" src="${frame}" alt="Frame ${index + 1}">`).join('')}
        <div class="play-button" onclick="playVideo()">
          <div class="play-icon"></div>
        </div>
        <div class="video-info">
          <div class="video-title">AI Generated Video</div>
          <div class="video-details">${prompt} • ${duration}s • ${style} style</div>
        </div>
      </div>
      
      <script>
        let currentFrame = 0;
        let isPlaying = false;
        let playInterval;
        const frames = document.querySelectorAll('.video-frame');
        const playButton = document.querySelector('.play-button');
        const totalFrames = ${frames.length};
        const frameDuration = ${frameDuration};
        
        function showFrame(index) {
          frames.forEach(frame => frame.classList.remove('active'));
          frames[index].classList.add('active');
        }
        
        function playVideo() {
          if (isPlaying) {
            stopVideo();
            return;
          }
          
          isPlaying = true;
          playButton.style.display = 'none';
          
          // Show first frame immediately
          showFrame(currentFrame);
          
          // Start cycling through frames
          playInterval = setInterval(() => {
            currentFrame = (currentFrame + 1) % totalFrames;
            showFrame(currentFrame);
          }, frameDuration);
          
          // Auto-stop after duration
          setTimeout(() => {
            stopVideo();
          }, parseInt('${duration}') * 1000);
        }
        
        function stopVideo() {
          isPlaying = false;
          if (playInterval) {
            clearInterval(playInterval);
          }
          playButton.style.display = 'flex';
        }
        
        // Initialize with first frame visible
        showFrame(0);
        
        // Auto-play after 1 second
        setTimeout(() => {
          playVideo();
        }, 1000);
      </script>
    </body>
    </html>
  `
  
  // Convert to data URL
  const blob = new Blob([videoHtml], { type: 'text/html' })
  return URL.createObjectURL(blob)
}

async function createFallbackVideo(prompt: string, duration: string, style: string, format: string): Promise<string> {
  // Generate a single representative image
  let imageUrl = ''
  
  try {
    const zai = await ZAI.create()
    const imageResponse = await zai.images.generations.create({
      prompt: `${prompt}, ${style} style, cinematic, high quality, video thumbnail`,
      size: '1024x1024'
    })
    
    if (imageResponse.data && imageResponse.data[0]?.base64) {
      imageUrl = imageResponse.data[0].base64
    }
  } catch (error) {
    console.error('Error generating thumbnail:', error)
  }
  
  // Create an animated video even with a single image
  const videoHtml = `
    <!DOCTYPE html>
    <html>
    <head>
      <meta charset="utf-8">
      <title>AI Generated Video: ${prompt}</title>
      <style>
        body { margin: 0; padding: 0; background: linear-gradient(45deg, #1a1a1a, #2d2d2d); display: flex; justify-content: center; align-items: center; height: 100vh; overflow: hidden; }
        .video-container { position: relative; width: 1024px; height: 1024px; max-width: 90vw; max-height: 90vh; }
        .main-image { width: 100%; height: 100%; object-fit: cover; animation: slowZoom 8s ease-in-out infinite alternate; }
        .overlay { position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: linear-gradient(to bottom, transparent 0%, rgba(0,0,0,0.3) 50%, rgba(0,0,0,0.7) 100%); }
        .video-info { position: absolute; bottom: 30px; left: 30px; right: 30px; color: white; text-align: center; font-family: Arial, sans-serif; }
        .video-title { font-size: 32px; font-weight: bold; margin-bottom: 10px; text-shadow: 2px 2px 4px rgba(0,0,0,0.8); animation: fadeInUp 2s ease-out; }
        .video-details { font-size: 18px; opacity: 0.9; text-shadow: 1px 1px 2px rgba(0,0,0,0.8); animation: fadeInUp 2s ease-out 0.5s both; }
        .video-badge { position: absolute; top: 30px; right: 30px; background: rgba(0,255,0,0.9); color: white; padding: 10px 20px; border-radius: 25px; font-weight: bold; animation: pulse 2s infinite; }
        .particles { position: absolute; top: 0; left: 0; width: 100%; height: 100%; pointer-events: none; }
        .particle { position: absolute; width: 4px; height: 4px; background: white; border-radius: 50%; animation: float 6s infinite ease-in-out; }
        
        @keyframes slowZoom {
          0% { transform: scale(1); }
          100% { transform: scale(1.1); }
        }
        
        @keyframes fadeInUp {
          0% { opacity: 0; transform: translateY(30px); }
          100% { opacity: 1; transform: translateY(0); }
        }
        
        @keyframes pulse {
          0%, 100% { opacity: 0.9; transform: scale(1); }
          50% { opacity: 1; transform: scale(1.05); }
        }
        
        @keyframes float {
          0%, 100% { transform: translateY(0px) rotate(0deg); opacity: 0.3; }
          50% { transform: translateY(-20px) rotate(180deg); opacity: 0.8; }
        }
      </style>
    </head>
    <body>
      <div class="video-container">
        ${imageUrl ? `<img class="main-image" src="${imageUrl}" alt="${prompt}">` : `
          <div style="width: 100%; height: 100%; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); display: flex; justify-content: center; align-items: center;">
            <div style="text-align: center; color: white;">
              <div style="font-size: 48px; margin-bottom: 20px;">🎬</div>
              <div style="font-size: 24px; font-weight: bold;">AI Generated Video</div>
            </div>
          </div>
        `}
        <div class="overlay"></div>
        <div class="particles">
          ${Array.from({length: 20}, (_, i) => `
            <div class="particle" style="left: ${Math.random() * 100}%; top: ${Math.random() * 100}%; animation-delay: ${Math.random() * 6}s;"></div>
          `).join('')}
        </div>
        <div class="video-info">
          <div class="video-title">${prompt}</div>
          <div class="video-details">${duration} seconds • ${style} style • AI Generated</div>
        </div>
        <div class="video-badge">AI VIDEO</div>
      </div>
      
      <script>
        // Add some interactive effects
        document.addEventListener('mousemove', (e) => {
          const particles = document.querySelectorAll('.particle');
          const x = e.clientX / window.innerWidth;
          const y = e.clientY / window.innerHeight;
          
          particles.forEach((particle, index) => {
            const speed = (index + 1) * 0.5;
            particle.style.transform = \`translate(\${x * speed}px, \${y * speed}px)\`;
          });
        });
      </script>
    </body>
    </html>
  `
  
  const blob = new Blob([videoHtml], { type: 'text/html' })
  return URL.createObjectURL(blob)
}

async function generateKeywords(zai: any, prompt: string, options: any) {
  const { domain = '', targetCountry = 'US' } = options
  
  const systemPrompt = `You are an SEO expert specializing in keyword research. Generate relevant keywords based on the given topic.`
  
  const userPrompt = `Generate 20 relevant SEO keywords for "${prompt}" for a website in ${targetCountry}. 
  Include a mix of informational, commercial, and transactional keywords. 
  For each keyword, provide estimated search volume (high/medium/low) and difficulty (easy/medium/hard).
  Format as JSON array with: {keyword, volume, difficulty, intent}`
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.5,
    max_tokens: 1500
  })
  
  const keywordsData = completion.choices[0]?.message?.content
  
  try {
    const keywords = JSON.parse(keywordsData || '[]')
    return {
      type: 'keywords',
      keywords,
      metadata: {
        prompt,
        domain,
        targetCountry
      }
    }
  } catch (error) {
    return {
      type: 'keywords',
      keywords: [],
      error: 'Failed to parse keywords'
    }
  }
}

async function optimizeContent(zai: any, prompt: string, options: any) {
  const { content, targetKeywords = [] } = options
  
  const systemPrompt = `You are an SEO expert. Analyze and optimize content for better search engine rankings and user engagement.`
  
  const userPrompt = `Analyze and optimize this content for SEO: "${content}". 
  ${targetKeywords.length > 0 ? `Target keywords: ${targetKeywords.join(', ')}` : ''}
  Provide:
  1. SEO score (0-100)
  2. Readability score (0-100)
  3. Keyword density analysis
  4. Specific recommendations for improvement
  5. Optimized version of the content`
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.3,
    max_tokens: 2500
  })
  
  const analysis = completion.choices[0]?.message?.content
  
  return {
    type: 'optimization',
    analysis,
    metadata: {
      originalContent: content,
      targetKeywords
    }
  }
}

async function analyzeTrends(zai: any, prompt: string, options: any) {
  const { platforms = ['instagram', 'twitter', 'tiktok', 'facebook', 'linkedin'], timeframe = '24h', category = 'general' } = options
  
  const systemPrompt = `You are a social media trends analyst with access to real-time data from all major social platforms. Your job is to identify trending topics, hashtags, and content patterns that are currently gaining traction.`
  
  const userPrompt = `Analyze current trending topics and content patterns across these platforms: ${platforms.join(', ')} for the ${timeframe} timeframe in the ${category} category.
  
  Provide a comprehensive analysis including:
  1. Top 10 trending topics/hashtags right now
  2. Engagement patterns (likes, comments, shares trends)
  3. Content formats that are performing well
  4. Best posting times and days
  5. Emerging trends to watch
  6. Platform-specific insights
  7. Viral content patterns
  8. Audience sentiment analysis
  
  Format as JSON with the following structure:
  {
    "trends": [
      {
        "topic": "trend name",
        "hashtag": "#hashtag",
        "platforms": ["instagram", "twitter"],
        "engagement": "high/medium/low",
        "growth": "+45%",
        "sentiment": "positive/neutral/negative",
        "description": "Brief description",
        "opportunity": "Content opportunity"
      }
    ],
    "insights": {
      "bestPostingTimes": ["9AM", "6PM"],
      "topFormats": ["short-video", "carousel"],
      "emergingTrends": ["trend1", "trend2"],
      "viralPatterns": ["pattern1", "pattern2"]
    },
    "recommendations": ["recommendation1", "recommendation2"]
  }`
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.4,
    max_tokens: 3000
  })
  
  const trendsData = completion.choices[0]?.message?.content
  
  try {
    const trends = JSON.parse(trendsData || '{}')
    return {
      type: 'trends',
      trends,
      metadata: {
        platforms,
        timeframe,
        category,
        analyzedAt: new Date().toISOString()
      }
    }
  } catch (error) {
    return {
      type: 'trends',
      trends: {
        trends: [],
        insights: {},
        recommendations: []
      },
      error: 'Failed to parse trends data'
    }
  }
}

async function generateTrendingContent(zai: any, prompt: string, options: any) {
  const { trend, platform = 'instagram', contentType = 'post', tone = 'engaging' } = options
  
  const systemPrompt = `You are a social media content expert who creates viral content based on current trends. You understand what makes content shareable and engaging across different platforms.`
  
  const userPrompt = `Create ${contentType} content for ${platform} based on this trend: "${trend}". The content should be ${tone} and optimized for maximum engagement.
  
  Generate:
  1. Compelling headline/caption
  2. Main content with hooks and emotional triggers
  3. Relevant hashtags (5-10)
  4. Call-to-action
  5. Visual content suggestions
  6. Best posting time recommendation
  7. Engagement boosters
  
  Format as JSON:
  {
    "headline": "Catchy headline",
    "content": "Main content text",
    "hashtags": ["#hashtag1", "#hashtag2"],
    "callToAction": "CTA text",
    "visualSuggestions": ["image idea 1", "video idea 2"],
    "bestPostingTime": "9AM EST",
    "engagementBoosters": ["ask question", "use emoji"],
    "platform": "${platform}",
    "contentType": "${contentType}"
  }`
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.7,
    max_tokens: 2000
  })
  
  const contentData = completion.choices[0]?.message?.content
  
  try {
    const content = JSON.parse(contentData || '{}')
    return {
      type: 'trending-content',
      content,
      metadata: {
        trend,
        platform,
        contentType,
        tone,
        generatedAt: new Date().toISOString()
      }
    }
  } catch (error) {
    return {
      type: 'trending-content',
      content: {},
      error: 'Failed to parse content data'
    }
  }
}