import RNFS from 'react-native-fs';

export class StorageService {
  private static readonly APP_DIR = `${RNFS.DocumentDirectoryPath}/social-media-trends`;
  private static readonly SERVER_DIR = `${StorageService.APP_DIR}/server`;
  private static readonly DATA_DIR = `${StorageService.APP_DIR}/data`;

  static async initialize(): Promise<void> {
    try {
      // Create necessary directories
      await RNFS.mkdir(StorageService.APP_DIR);
      await RNFS.mkdir(StorageService.SERVER_DIR);
      await RNFS.mkdir(StorageService.DATA_DIR);
      
      console.log('Storage directories created successfully');
    } catch (error) {
      console.error('Failed to initialize storage:', error);
      throw error;
    }
  }

  static async saveData(key: string, data: any): Promise<void> {
    try {
      const filePath = `${StorageService.DATA_DIR}/${key}.json`;
      const jsonData = JSON.stringify(data, null, 2);
      await RNFS.writeFile(filePath, jsonData, 'utf8');
      console.log(`Data saved to ${filePath}`);
    } catch (error) {
      console.error(`Failed to save data for key ${key}:`, error);
      throw error;
    }
  }

  static async loadData(key: string): Promise<any> {
    try {
      const filePath = `${StorageService.DATA_DIR}/${key}.json`;
      const jsonData = await RNFS.readFile(filePath, 'utf8');
      return JSON.parse(jsonData);
    } catch (error) {
      console.error(`Failed to load data for key ${key}:`, error);
      return null;
    }
  }

  static async saveServerFiles(): Promise<void> {
    try {
      // Copy Next.js build files to app directory
      const serverSourcePath = `${RNFS.DocumentDirectoryPath}/../assets/server`;
      const serverDestPath = StorageService.SERVER_DIR;

      if (await RNFS.exists(serverSourcePath)) {
        await StorageService.copyDirectory(serverSourcePath, serverDestPath);
        console.log('Server files copied successfully');
      }
    } catch (error) {
      console.error('Failed to save server files:', error);
      throw error;
    }
  }

  private static async copyDirectory(source: string, destination: string): Promise<void> {
    try {
      const exists = await RNFS.exists(destination);
      if (!exists) {
        await RNFS.mkdir(destination);
      }

      const items = await RNFS.readDir(source);
      
      for (const item of items) {
        const destPath = `${destination}/${item.name}`;
        
        if (item.isDirectory()) {
          await StorageService.copyDirectory(item.path, destPath);
        } else {
          await RNFS.copyFile(item.path, destPath);
        }
      }
    } catch (error) {
      console.error('Failed to copy directory:', error);
      throw error;
    }
  }

  static async getServerPath(): Promise<string> {
    return StorageService.SERVER_DIR;
  }

  static async clearCache(): Promise<void> {
    try {
      const cacheDir = `${StorageService.APP_DIR}/cache`;
      if (await RNFS.exists(cacheDir)) {
        await RNFS.unlink(cacheDir);
      }
      await RNFS.mkdir(cacheDir);
      console.log('Cache cleared successfully');
    } catch (error) {
      console.error('Failed to clear cache:', error);
    }
  }

  static async getStorageInfo(): Promise<{
    totalSpace: number;
    freeSpace: number;
    usedSpace: number;
  }> {
    try {
      const totalSpace = await RNFS.getFSInfo();
      const serverSize = await StorageService.getDirectorySize(StorageService.SERVER_DIR);
      const dataSize = await StorageService.getDirectorySize(StorageService.DATA_DIR);
      
      return {
        totalSpace: totalSpace.totalSpace || 0,
        freeSpace: totalSpace.freeSpace || 0,
        usedSpace: serverSize + dataSize,
      };
    } catch (error) {
      console.error('Failed to get storage info:', error);
      return {
        totalSpace: 0,
        freeSpace: 0,
        usedSpace: 0,
      };
    }
  }

  private static async getDirectorySize(directory: string): Promise<number> {
    try {
      if (!(await RNFS.exists(directory))) {
        return 0;
      }

      const items = await RNFS.readDir(directory);
      let totalSize = 0;

      for (const item of items) {
        if (item.isDirectory()) {
          totalSize += await StorageService.getDirectorySize(item.path);
        } else {
          totalSize += item.size;
        }
      }

      return totalSize;
    } catch (error) {
      console.error(`Failed to get directory size for ${directory}:`, error);
      return 0;
    }
  }
}