import { NativeModules, Platform } from 'react-native';

const { ServerModule } = NativeModules;

export class NodeServer {
  private static isRunning = false;
  private static serverUrl = '';

  static async startServer(): Promise<string> {
    try {
      if (this.isRunning) {
        return this.serverUrl;
      }

      console.log('Starting Node.js server on Android...');

      if (Platform.OS === 'android') {
        // Use native Android module to start Node.js server
        const url = await ServerModule.startNodeServer();
        this.serverUrl = url;
        this.isRunning = true;
        
        console.log('Node.js server started at:', url);
        return url;
      } else {
        throw new Error('Platform not supported');
      }
    } catch (error) {
      console.error('Failed to start Node.js server:', error);
      throw error;
    }
  }

  static async stopServer(): Promise<void> {
    try {
      if (!this.isRunning) {
        return;
      }

      console.log('Stopping Node.js server...');

      if (Platform.OS === 'android') {
        await ServerModule.stopNodeServer();
      }

      this.isRunning = false;
      this.serverUrl = '';
      
      console.log('Node.js server stopped');
    } catch (error) {
      console.error('Failed to stop Node.js server:', error);
      throw error;
    }
  }

  static isServerRunning(): boolean {
    return this.isRunning;
  }

  static getServerUrl(): string {
    return this.serverUrl;
  }

  static async checkServerHealth(): Promise<boolean> {
    try {
      if (!this.isRunning || !this.serverUrl) {
        return false;
      }

      const response = await fetch(`${this.serverUrl}/api/health`, {
        method: 'GET',
        timeout: 5000,
      });

      return response.ok;
    } catch (error) {
      console.error('Server health check failed:', error);
      return false;
    }
  }
}