package com.socialmediatrends;

import android.content.Context;
import android.os.AsyncTask;
import android.os.Build;
import android.util.Log;

import androidx.annotation.NonNull;

import com.facebook.react.bridge.Promise;
import com.facebook.react.bridge.ReactApplicationContext;
import com.facebook.react.bridge.ReactContextBaseJavaModule;
import com.facebook.react.bridge.ReactMethod;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

public class ServerModule extends ReactContextBaseJavaModule {
    private static final String TAG = "ServerModule";
    private static final String SERVER_PORT = "3000";
    private NodeServerTask serverTask;
    private String serverUrl;

    public ServerModule(ReactApplicationContext reactContext) {
        super(reactContext);
    }

    @Override
    public String getName() {
        return "ServerModule";
    }

    @ReactMethod
    public void startNodeServer(Promise promise) {
        try {
            Context context = getReactApplicationContext();
            
            // Extract server files from assets
            extractServerFiles(context);
            
            // Start Node.js server
            serverTask = new NodeServerTask(context);
            serverTask.execute();
            
            // Wait a moment for server to start
            new android.os.Handler().postDelayed(() -> {
                serverUrl = "http://localhost:" + SERVER_PORT;
                Log.d(TAG, "Server started at: " + serverUrl);
                promise.resolve(serverUrl);
            }, 3000);
            
        } catch (Exception e) {
            Log.e(TAG, "Failed to start server", e);
            promise.reject("SERVER_START_ERROR", e.getMessage());
        }
    }

    @ReactMethod
    public void stopNodeServer(Promise promise) {
        try {
            if (serverTask != null) {
                serverTask.cancel(true);
                serverTask = null;
            }
            Log.d(TAG, "Server stopped");
            promise.resolve("Server stopped");
        } catch (Exception e) {
            Log.e(TAG, "Failed to stop server", e);
            promise.reject("SERVER_STOP_ERROR", e.getMessage());
        }
    }

    private void extractServerFiles(Context context) throws IOException {
        File serverDir = new File(context.getFilesDir(), "server");
        if (!serverDir.exists()) {
            serverDir.mkdirs();
        }

        // Copy server files from assets
        copyAssetFolder(context.getAssets(), "server", serverDir.getAbsolutePath());
        
        // Copy node_modules if they exist
        File nodeModulesDir = new File(context.getFilesDir(), "node_modules");
        if (!nodeModulesDir.exists()) {
            copyAssetFolder(context.getAssets(), "node_modules", nodeModulesDir.getAbsolutePath());
        }

        Log.d(TAG, "Server files extracted to: " + serverDir.getAbsolutePath());
    }

    private void copyAssetFolder(android.content.res.AssetManager assetManager, 
            String fromAssetPath, String toPath) throws IOException {
        String[] files = assetManager.list(fromAssetPath);
        if (files == null) return;

        File dir = new File(toPath);
        if (!dir.exists()) {
            dir.mkdirs();
        }

        for (String file : files) {
            String fromAssetFile = fromAssetPath + "/" + file;
            String toFile = toPath + "/" + file;
            
            try (InputStream in = assetManager.open(fromAssetFile)) {
                if (assetManager.list(fromAssetFile) != null) {
                    // It's a directory
                    copyAssetFolder(assetManager, fromAssetFile, toFile);
                } else {
                    // It's a file
                    copyFile(in, toFile);
                }
            }
        }
    }

    private void copyFile(InputStream in, String toFile) throws IOException {
        OutputStream out = new FileOutputStream(toFile);
        byte[] buffer = new byte[1024];
        int read;
        while ((read = in.read(buffer)) != -1) {
            out.write(buffer, 0, read);
        }
        in.close();
        out.flush();
        out.close();
    }

    private static class NodeServerTask extends AsyncTask<Void, String, String> {
        private Context context;
        private Process nodeProcess;

        public NodeServerTask(Context context) {
            this.context = context;
        }

        @Override
        protected String doInBackground(Void... voids) {
            try {
                File serverDir = new File(context.getFilesDir(), "server");
                File packageJson = new File(serverDir, "package.json");
                
                if (!packageJson.exists()) {
                    Log.e(TAG, "package.json not found in server directory");
                    return "package.json not found";
                }

                // Set up environment variables
                String[] env = {
                    "NODE_ENV=production",
                    "PORT=" + SERVER_PORT,
                    "HOST=0.0.0.0",
                    "NODE_PATH=" + context.getFilesDir().getAbsolutePath() + "/node_modules"
                };

                // Build command to start Node.js server
                ProcessBuilder pb = new ProcessBuilder();
                
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    // Android 10+ - use bundled Node.js binary
                    File nodeBinary = new File(context.getApplicationInfo().nativeLibraryDir, "libnode.so");
                    if (nodeBinary.exists()) {
                        pb.command(nodeBinary.getAbsolutePath(), "server.js");
                    } else {
                        Log.e(TAG, "Node.js binary not found");
                        return "Node.js binary not found";
                    }
                } else {
                    // Older Android - try system node if available
                    pb.command("node", "server.js");
                }
                
                pb.directory(serverDir);
                pb.environment().putAll(System.getenv());
                for (String envVar : env) {
                    String[] parts = envVar.split("=", 2);
                    if (parts.length == 2) {
                        pb.environment().put(parts[0], parts[1]);
                    }
                }

                // Redirect error stream to output stream
                pb.redirectErrorStream(true);

                Log.d(TAG, "Starting Node.js server with command: " + pb.command());
                Log.d(TAG, "Working directory: " + pb.directory());
                Log.d(TAG, "Environment: " + pb.environment());

                nodeProcess = pb.start();

                // Read server output
                java.io.BufferedReader reader = new java.io.BufferedReader(
                    new java.io.InputStreamReader(nodeProcess.getInputStream())
                );

                String line;
                while ((line = reader.readLine()) != null && !isCancelled()) {
                    Log.d(TAG, "Server: " + line);
                    publishProgress(line);
                }

                int exitCode = nodeProcess.waitFor();
                Log.d(TAG, "Server process exited with code: " + exitCode);
                
                return "Server exited with code: " + exitCode;

            } catch (Exception e) {
                Log.e(TAG, "Error running Node.js server", e);
                return "Error: " + e.getMessage();
            }
        }

        @Override
        protected void onProgressUpdate(String... values) {
            super.onProgressUpdate(values);
            for (String message : values) {
                Log.d(TAG, "Server log: " + message);
            }
        }

        @Override
        protected void onCancelled(String result) {
            super.onCancelled(result);
            if (nodeProcess != null) {
                nodeProcess.destroy();
            }
            Log.d(TAG, "Server task cancelled");
        }

        @Override
        protected void onPostExecute(String result) {
            super.onPostExecute(result);
            Log.d(TAG, "Server task completed: " + result);
        }
    }
}