const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const compression = require('compression');
const morgan = require('morgan');
const path = require('path');
const http = require('http');

// Import ZAI SDK
const ZAI = require('z-ai-web-dev-sdk');

const app = express();
const server = http.createServer(app);

// Middleware
app.use(helmet({
  contentSecurityPolicy: false,
  crossOriginEmbedderPolicy: false,
}));
app.use(compression());
app.use(cors());
app.use(morgan('combined'));
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Serve static files from the Next.js build
app.use(express.static(path.join(__dirname, 'public')));

// API Routes
app.get('/api/health', (req, res) => {
  res.json({ 
    status: 'ok', 
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    memory: process.memoryUsage()
  });
});

app.post('/api/ai', async (req, res) => {
  try {
    const { type, prompt, options = {} } = req.body;
    
    const zai = await ZAI.create();
    
    let response;
    
    switch (type) {
      case 'content':
        response = await generateContent(zai, prompt, options);
        break;
      case 'image':
        response = await generateImage(zai, prompt, options);
        break;
      case 'keywords':
        response = await generateKeywords(zai, prompt, options);
        break;
      case 'optimize':
        response = await optimizeContent(zai, prompt, options);
        break;
      case 'trends':
        response = await analyzeTrends(zai, prompt, options);
        break;
      case 'trending-content':
        response = await generateTrendingContent(zai, prompt, options);
        break;
      default:
        return res.status(400).json({ error: 'Invalid AI type' });
    }
    
    res.json(response);
  } catch (error) {
    console.error('Error in AI generation:', error);
    res.status(500).json({ error: 'Failed to generate content' });
  }
});

// AI Functions
async function generateContent(zai, prompt, options) {
  const { contentType = 'blog post', tone = 'professional', length = 'medium', targetKeywords = [] } = options;
  
  const systemPrompt = `You are a professional content writer and SEO expert. Create high-quality, engaging content that performs well on social media and search engines.`;
  
  const userPrompt = `Generate a ${contentType} about "${prompt}" with a ${tone} tone. The content should be approximately ${length} words. 
  ${targetKeywords.length > 0 ? `Include these keywords naturally: ${targetKeywords.join(', ')}` : ''}
  Make it SEO-optimized, engaging, and include a call-to-action.`;
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.7,
    max_tokens: 2000
  });
  
  const content = completion.choices[0]?.message?.content;
  
  return {
    type: 'content',
    content,
    metadata: {
      contentType,
      tone,
      length,
      targetKeywords
    }
  };
}

async function generateImage(zai, prompt, options) {
  const { size = '512x512', style = 'realistic' } = options;
  
  const enhancedPrompt = `${prompt}, ${style} style, high quality, professional, social media ready`;
  
  const imageResponse = await zai.images.generations.create({
    prompt: enhancedPrompt,
    size
  });
  
  const imageBase64 = imageResponse.data[0]?.base64;
  
  return {
    type: 'image',
    image: imageBase64,
    metadata: {
      prompt,
      size,
      style
    }
  };
}

async function generateKeywords(zai, prompt, options) {
  const { domain = '', targetCountry = 'US' } = options;
  
  const systemPrompt = `You are an SEO expert specializing in keyword research. Generate relevant keywords based on the given topic.`;
  
  const userPrompt = `Generate 20 relevant SEO keywords for "${prompt}" for a website in ${targetCountry}. 
  Include a mix of informational, commercial, and transactional keywords. 
  For each keyword, provide estimated search volume (high/medium/low) and difficulty (easy/medium/hard).
  Format as JSON array with: {keyword, volume, difficulty, intent}`;
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.5,
    max_tokens: 1500
  });
  
  const keywordsData = completion.choices[0]?.message?.content;
  
  try {
    const keywords = JSON.parse(keywordsData || '[]');
    return {
      type: 'keywords',
      keywords,
      metadata: {
        prompt,
        domain,
        targetCountry
      }
    };
  } catch (error) {
    return {
      type: 'keywords',
      keywords: [],
      error: 'Failed to parse keywords'
    };
  }
}

async function optimizeContent(zai, prompt, options) {
  const { content, targetKeywords = [] } = options;
  
  const systemPrompt = `You are an SEO expert. Analyze and optimize content for better search engine rankings and user engagement.`;
  
  const userPrompt = `Analyze and optimize this content for SEO: "${content}". 
  ${targetKeywords.length > 0 ? `Target keywords: ${targetKeywords.join(', ')}` : ''}
  Provide:
  1. SEO score (0-100)
  2. Readability score (0-100)
  3. Keyword density analysis
  4. Specific recommendations for improvement
  5. Optimized version of the content`;
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.3,
    max_tokens: 2500
  });
  
  const analysis = completion.choices[0]?.message?.content;
  
  return {
    type: 'optimization',
    analysis,
    metadata: {
      originalContent: content,
      targetKeywords
    }
  };
}

async function analyzeTrends(zai, prompt, options) {
  const { platforms = ['instagram', 'twitter', 'tiktok', 'facebook', 'linkedin'], timeframe = '24h', category = 'general' } = options;
  
  const systemPrompt = `You are a social media trends analyst with access to real-time data from all major social platforms. Your job is to identify trending topics, hashtags, and content patterns that are currently gaining traction.`;
  
  const userPrompt = `Analyze current trending topics and content patterns across these platforms: ${platforms.join(', ')} for the ${timeframe} timeframe in the ${category} category.
  
  Provide a comprehensive analysis including:
  1. Top 10 trending topics/hashtags right now
  2. Engagement patterns (likes, comments, shares trends)
  3. Content formats that are performing well
  4. Best posting times and days
  5. Emerging trends to watch
  6. Platform-specific insights
  7. Viral content patterns
  8. Audience sentiment analysis
  
  Format as JSON with the following structure:
  {
    "trends": [
      {
        "topic": "trend name",
        "hashtag": "#hashtag",
        "platforms": ["instagram", "twitter"],
        "engagement": "high/medium/low",
        "growth": "+45%",
        "sentiment": "positive/neutral/negative",
        "description": "Brief description",
        "opportunity": "Content opportunity"
      }
    ],
    "insights": {
      "bestPostingTimes": ["9AM", "6PM"],
      "topFormats": ["short-video", "carousel"],
      "emergingTrends": ["trend1", "trend2"],
      "viralPatterns": ["pattern1", "pattern2"]
    },
    "recommendations": ["recommendation1", "recommendation2"]
  }`;
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.4,
    max_tokens: 3000
  });
  
  const trendsData = completion.choices[0]?.message?.content;
  
  try {
    const trends = JSON.parse(trendsData || '{}');
    return {
      type: 'trends',
      trends,
      metadata: {
        platforms,
        timeframe,
        category,
        analyzedAt: new Date().toISOString()
      }
    };
  } catch (error) {
    return {
      type: 'trends',
      trends: {
        trends: [],
        insights: {},
        recommendations: []
      },
      error: 'Failed to parse trends data'
    };
  }
}

async function generateTrendingContent(zai, prompt, options) {
  const { trend, platform = 'instagram', contentType = 'post', tone = 'engaging' } = options;
  
  const systemPrompt = `You are a social media content expert who creates viral content based on current trends. You understand what makes content shareable and engaging across different platforms.`;
  
  const userPrompt = `Create ${contentType} content for ${platform} based on this trend: "${trend}". The content should be ${tone} and optimized for maximum engagement.
  
  Generate:
  1. Compelling headline/caption
  2. Main content with hooks and emotional triggers
  3. Relevant hashtags (5-10)
  4. Call-to-action
  5. Visual content suggestions
  6. Best posting time recommendation
  7. Engagement boosters
  
  Format as JSON:
  {
    "headline": "Catchy headline",
    "content": "Main content text",
    "hashtags": ["#hashtag1", "#hashtag2"],
    "callToAction": "CTA text",
    "visualSuggestions": ["image idea 1", "video idea 2"],
    "bestPostingTime": "9AM EST",
    "engagementBoosters": ["ask question", "use emoji"],
    "platform": "${platform}",
    "contentType": "${contentType}"
  }`;
  
  const completion = await zai.chat.completions.create({
    messages: [
      { role: 'system', content: systemPrompt },
      { role: 'user', content: userPrompt }
    ],
    temperature: 0.7,
    max_tokens: 2000
  });
  
  const contentData = completion.choices[0]?.message?.content;
  
  try {
    const content = JSON.parse(contentData || '{}');
    return {
      type: 'trending-content',
      content,
      metadata: {
        trend,
        platform,
        contentType,
        tone,
        generatedAt: new Date().toISOString()
      }
    };
  } catch (error) {
    return {
      type: 'trending-content',
      content: {},
      error: 'Failed to parse content data'
    };
  }
}

// Serve Next.js app for all other routes
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// Start server
const PORT = process.env.PORT || 3000;
const HOST = process.env.HOST || '0.0.0.0';

server.listen(PORT, HOST, () => {
  console.log(`🚀 Social Media Trends Server running on http://${HOST}:${PORT}`);
  console.log(`📱 Mobile server ready for WebView connection`);
  console.log(`🔧 Environment: ${process.env.NODE_ENV || 'development'}`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM received, shutting down gracefully');
  server.close(() => {
    console.log('Process terminated');
  });
});

process.on('SIGINT', () => {
  console.log('SIGINT received, shutting down gracefully');
  server.close(() => {
    console.log('Process terminated');
  });
});