import React, { useEffect, useState } from 'react';
import {
  SafeAreaView,
  StyleSheet,
  View,
  Text,
  TouchableOpacity,
  PermissionsAndroid,
  Platform,
  Alert,
  StatusBar,
  NativeModules,
} from 'react-native';
import { WebView } from 'react-native-webview';
import SplashScreen from 'react-native-splash-screen';
import { NodeServer } from './src/services/NodeServer';
import { StorageService } from './src/services/StorageService';

const { ServerModule } = NativeModules;

export default function App() {
  const [serverUrl, setServerUrl] = useState<string>('');
  const [isServerRunning, setIsServerRunning] = useState<boolean>(false);
  const [serverStatus, setServerStatus] = useState<string>('Starting server...');
  const [error, setError] = useState<string>('');

  useEffect(() => {
    // Hide splash screen
    SplashScreen.hide();
    
    // Initialize app
    initializeApp();
    
    return () => {
      // Cleanup server on unmount
      NodeServer.stopServer();
    };
  }, []);

  const initializeApp = async () => {
    try {
      // Request necessary permissions
      await requestPermissions();
      
      // Start the Node.js server
      await startLocalServer();
      
    } catch (err) {
      console.error('App initialization error:', err);
      setError('Failed to initialize app: ' + err.message);
    }
  };

  const requestPermissions = async () => {
    if (Platform.OS === 'android') {
      try {
        const granted = await PermissionsAndroid.requestMultiple([
          PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE,
          PermissionsAndroid.PERMISSIONS.READ_EXTERNAL_STORAGE,
          PermissionsAndroid.PERMISSIONS.INTERNET,
          PermissionsAndroid.PERMISSIONS.ACCESS_NETWORK_STATE,
        ]);
        
        console.log('Permissions granted:', granted);
      } catch (err) {
        console.warn('Permission request failed:', err);
      }
    }
  };

  const startLocalServer = async () => {
    try {
      setServerStatus('Initializing Node.js server...');
      
      // Start the built-in Node.js server
      const url = await NodeServer.startServer();
      
      setServerUrl(url);
      setIsServerRunning(true);
      setServerStatus('Server running successfully!');
      
      console.log('Server started at:', url);
      
    } catch (err) {
      console.error('Failed to start server:', err);
      setError('Failed to start server: ' + err.message);
      setServerStatus('Server failed to start');
    }
  };

  const restartServer = async () => {
    setIsServerRunning(false);
    setServerUrl('');
    setError('');
    setServerStatus('Restarting server...');
    
    try {
      await NodeServer.stopServer();
      await startLocalServer();
    } catch (err) {
      console.error('Failed to restart server:', err);
      setError('Failed to restart server: ' + err.message);
    }
  };

  const renderWebView = () => {
    if (!isServerRunning || !serverUrl) {
      return null;
    }

    return (
      <WebView
        source={{ uri: serverUrl }}
        style={styles.webView}
        javaScriptEnabled={true}
        domStorageEnabled={true}
        allowFileAccess={true}
        allowUniversalAccessFromFileURLs={true}
        mixedContentMode="compatibility"
        onLoadStart={() => setServerStatus('Loading web application...')}
        onLoadEnd={() => setServerStatus('Application ready!')}
        onError={(syntheticEvent) => {
          const { nativeEvent } = syntheticEvent;
          console.error('WebView error: ', nativeEvent);
          setError('WebView error: ' + nativeEvent.description);
        }}
        onHttpError={(syntheticEvent) => {
          const { nativeEvent } = syntheticEvent;
          console.error('WebView HTTP error: ', nativeEvent);
          setError('HTTP error: ' + nativeEvent.description);
        }}
        injectedJavaScript={`
          // Inject mobile-specific styles and functionality
          (function() {
            // Add mobile viewport meta tag
            const viewport = document.querySelector('meta[name="viewport"]');
            if (!viewport) {
              const meta = document.createElement('meta');
              meta.name = 'viewport';
              meta.content = 'width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no';
              document.head.appendChild(meta);
            }
            
            // Add mobile app detection
            window.isMobileApp = true;
            window.isAndroid = true;
            
            // Override console.log to send to React Native
            const originalLog = console.log;
            console.log = function(...args) {
              originalLog.apply(console, args);
              window.ReactNativeWebView.postMessage(JSON.stringify({
                type: 'console',
                data: args
              }));
            };
            
            // Notify React Native when page is ready
            window.ReactNativeWebView.postMessage(JSON.stringify({
              type: 'ready',
              url: window.location.href
            }));
          })();
        `}
        onMessage={(event) => {
          try {
            const data = JSON.parse(event.nativeEvent.data);
            console.log('WebView message:', data);
            
            if (data.type === 'ready') {
              setServerStatus('Application loaded successfully!');
            }
          } catch (err) {
            console.error('Failed to parse WebView message:', err);
          }
        }}
      />
    );
  };

  const renderLoadingScreen = () => (
    <View style={styles.loadingContainer}>
      <View style={styles.loadingCard}>
        <Text style={styles.loadingTitle}>Social Media Trends</Text>
        <Text style={styles.loadingSubtitle}>AI-Powered Trend Analysis</Text>
        
        <View style={styles.statusContainer}>
          <Text style={styles.statusText}>{serverStatus}</Text>
        </View>
        
        {error && (
          <View style={styles.errorContainer}>
            <Text style={styles.errorText}>{error}</Text>
          </View>
        )}
        
        {!isServerRunning && (
          <TouchableOpacity style={styles.retryButton} onPress={restartServer}>
            <Text style={styles.retryButtonText}>Retry</Text>
          </TouchableOpacity>
        )}
      </View>
    </View>
  );

  return (
    <SafeAreaView style={styles.container}>
      <StatusBar barStyle="light-content" backgroundColor="#1f2937" />
      
      {isServerRunning ? (
        <View style={styles.container}>
          {renderWebView()}
          
          {/* Server status overlay for debugging */}
          {__DEV__ && (
            <View style={styles.debugOverlay}>
              <Text style={styles.debugText}>Server: {serverUrl}</Text>
              <TouchableOpacity style={styles.debugButton} onPress={restartServer}>
                <Text style={styles.debugButtonText}>Restart</Text>
              </TouchableOpacity>
            </View>
          )}
        </View>
      ) : (
        renderLoadingScreen()
      )}
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#1f2937',
  },
  webView: {
    flex: 1,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#1f2937',
    padding: 20,
  },
  loadingCard: {
    backgroundColor: '#374151',
    borderRadius: 16,
    padding: 30,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  loadingTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#ffffff',
    marginBottom: 8,
    textAlign: 'center',
  },
  loadingSubtitle: {
    fontSize: 16,
    color: '#9ca3af',
    marginBottom: 30,
    textAlign: 'center',
  },
  statusContainer: {
    backgroundColor: '#1f2937',
    borderRadius: 8,
    padding: 15,
    marginBottom: 20,
    minWidth: 200,
  },
  statusText: {
    color: '#10b981',
    fontSize: 14,
    textAlign: 'center',
    fontWeight: '500',
  },
  errorContainer: {
    backgroundColor: '#991b1b',
    borderRadius: 8,
    padding: 15,
    marginBottom: 20,
    minWidth: 200,
  },
  errorText: {
    color: '#ffffff',
    fontSize: 14,
    textAlign: 'center',
  },
  retryButton: {
    backgroundColor: '#3b82f6',
    borderRadius: 8,
    paddingHorizontal: 24,
    paddingVertical: 12,
  },
  retryButtonText: {
    color: '#ffffff',
    fontSize: 16,
    fontWeight: '600',
  },
  debugOverlay: {
    position: 'absolute',
    top: 50,
    right: 10,
    backgroundColor: 'rgba(0,0,0,0.8)',
    borderRadius: 8,
    padding: 10,
    zIndex: 1000,
  },
  debugText: {
    color: '#ffffff',
    fontSize: 12,
    marginBottom: 5,
  },
  debugButton: {
    backgroundColor: '#3b82f6',
    borderRadius: 4,
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  debugButtonText: {
    color: '#ffffff',
    fontSize: 12,
  },
});